//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.temporalrelation;

import de.ugoe.cs.autoquest.tasktrees.taskequality.TaskEquality;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.usageprofiles.SymbolComparator;
import de.ugoe.cs.autoquest.usageprofiles.SymbolMap;
import de.ugoe.cs.autoquest.usageprofiles.SymbolStrategy;

/**
 * <p>
 * concrete implementation of a symbol strategy required in the creation of a
 * {@link de.ugoe.cs.autoquest.usageprofiles.Trie}. The strategy can be parameterized with a
 * considered task equality. It uses task instance comparators to perform task comparison.
 * It creates task specific symbol maps, which are {@link TaskSymbolIdentityMap} and
 * {@link TaskSymbolBucketedMap} depending on the level of considered task equality.
 * </p>
 * 
 * @author Patrick Harms
 */
public class TaskHandlingStrategy implements SymbolStrategy<ITask> {
    
    /**  */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * the level of task equality considered in this task handling strategy
     * </p>
     */
    private TaskEquality consideredEquality;

    /**
     * <p>
     * the comparator used for task comparisons
     * </p>
     */
    private TaskComparator comparator;

    /**
     * <p>
     * initializes this strategy with a task equality to be considered for task comparisons
     * </p>
     *
     * @param consideredEquality the task equality to be considered for task comparisons
     */
    public TaskHandlingStrategy(TaskEquality consideredEquality) {
        this.consideredEquality = consideredEquality;
        
        if (this.consideredEquality == TaskEquality.IDENTICAL) {
            comparator = new TaskIdentityComparator();
        }
        else {
            comparator = new TaskComparator(this.consideredEquality);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolStrategy#getSymbolComparator()
     */
    @Override
    public SymbolComparator<ITask> getSymbolComparator() {
        return comparator;
    }

    /**
     * <p>
     * convenience method to have a correctly typed return value as alternative to
     * {@link #getSymbolComparator()};
     * </p>
     */
    public TaskComparator getTaskComparator() {
        return comparator;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolStrategy#createSymbolMap()
     */
    @Override
    public <V> SymbolMap<ITask, V> createSymbolMap() {
        if (consideredEquality == TaskEquality.IDENTICAL) {
            return new TaskSymbolIdentityMap<V>();
        }
        else {
            return new TaskSymbolBucketedMap<V>(comparator);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.usageprofiles.SymbolStrategy#copySymbolMap(SymbolMap)
     */
    @Override
    public <V> SymbolMap<ITask, V> copySymbolMap(SymbolMap<ITask, V> other) {
        if (consideredEquality == TaskEquality.IDENTICAL) {
            return new TaskSymbolIdentityMap<V>(other);
        }
        else {
            return new TaskSymbolBucketedMap<V>(comparator);
        }
    }

}
