//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.IButton;
import de.ugoe.cs.autoquest.eventcore.guimodel.ICheckBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IComboBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IListBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IMenuButton;
import de.ugoe.cs.autoquest.eventcore.guimodel.ITextArea;
import de.ugoe.cs.autoquest.eventcore.guimodel.ITextField;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class UnusedGUIElementsRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        GUIModel guiModel = getGUIModel(taskModel);
        Set<IGUIElement> allGUIElements = getAllGUIElements(guiModel);
        Set<IGUIElement> usedGUIElements = getUsedGUIElements(taskModel);
        List<IGUIElement> unusedGUIElements = getUnusedGUIElements(usedGUIElements, allGUIElements);
        handleUnusedGUIElements(unusedGUIElements, allGUIElements, results);

        return results;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param unusedGUIElements
     * @param results
     */
    private void handleUnusedGUIElements(List<IGUIElement>         unusedGUIElements,
                                         Set<IGUIElement>          allGUIElements,
                                         UsabilityEvaluationResult results)
    {
        int ratio = 1000 * unusedGUIElements.size() / allGUIElements.size();
        
        UsabilitySmellIntensity severity = UsabilitySmellIntensity.getIntensity(ratio);

        if (severity != null) {
            Map<String, Object> parameters = new HashMap<String, Object>();

            parameters.put("ratio", ratio / 10);
            parameters.put("noOfUnused", unusedGUIElements.size());
            parameters.put("noOfAll", allGUIElements.size());
            parameters.put("unusedGuiElements", unusedGUIElements);
            
            results.addSmell
                (severity, UsabilitySmellDescription.UNUSED_GUI_ELEMENTS, parameters);
        }
     }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     * @return
     */
    private GUIModel getGUIModel(ITaskModel taskModel) {
        for (ITask task : taskModel.getTasks()) {
            if (task instanceof IEventTask) {
                for (ITaskInstance instance : task.getInstances()) {
                    Event event = ((IEventTaskInstance) instance).getEvent();
                    
                    if (event.getTarget() instanceof IGUIElement) {
                        return ((IGUIElement) event.getTarget()).getGUIModel();
                    }
                }
            }
        }
        
        return null;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private Set<IGUIElement> getUsedGUIElements(ITaskModel taskModel) {
        Set<IGUIElement> usedGUIElements = new HashSet<IGUIElement>();
        
        for (ITask task : taskModel.getTasks()) {
            if (task instanceof IEventTask) {
                for (ITaskInstance instance : task.getInstances()) {
                    Event event = ((IEventTaskInstance) instance).getEvent();
                    
                    if (event.getTarget() instanceof IGUIElement) {
                        usedGUIElements.add((IGUIElement) event.getTarget());
                    }
                }
            }
        }
        
        return usedGUIElements;
    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private Set<IGUIElement> getAllGUIElements(GUIModel guiModel) {
        Set<IGUIElement> allGUIElements = new HashSet<IGUIElement>();
        
        if (guiModel != null) {
            GUIModel.Traverser traverser = guiModel.getTraverser();

            IGUIElement currentGUIElement = null;
            do {
                if (traverser.hasFirstChild()) {
                    currentGUIElement = traverser.firstChild();
                }
                else if (traverser.hasNextSibling()) {
                    currentGUIElement = traverser.nextSibling();
                }
                else {
                    while (currentGUIElement != null) {
                        currentGUIElement = traverser.parent();
                        if (traverser.hasNextSibling()) {
                            currentGUIElement = traverser.nextSibling();
                            break;
                        }
                    }
                }

                if (isRelevant(currentGUIElement)) {
                    allGUIElements.add(currentGUIElement);
                }
            }
            while (currentGUIElement != null);
        }
        
        return allGUIElements;
    }
    
    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private List<IGUIElement> getUnusedGUIElements(Set<IGUIElement> usedGUIElements,
                                                   Set<IGUIElement> allGUIElements)
    {
        List<IGUIElement> unusedGUIElements = new LinkedList<IGUIElement>();
        for (IGUIElement currentGUIElement : allGUIElements) {
            if (isRelevant(currentGUIElement) &&
                !belongsToUsedGUIElements(currentGUIElement, usedGUIElements))
            {
                unusedGUIElements.add(currentGUIElement);
            }
        }
        
        return unusedGUIElements;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param currentGUIElement
     * @param usedGUIElements
     * @return
     */
    private boolean belongsToUsedGUIElements(IGUIElement      relevantGUIElement,
                                             Set<IGUIElement> usedGUIElements)
    {
        if (usedGUIElements.contains(relevantGUIElement)) {
            return true;
        }
        else {
            // in some cases, the events are recorded for the children of the relevant GUI elements
            // therefore, check the children, as well.
            List<IGUIElement> children =
                relevantGUIElement.getGUIModel().getChildren(relevantGUIElement);
            
            if (children != null) {
                for (IGUIElement child : children) {
                    if (belongsToUsedGUIElements(child, usedGUIElements)) {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param currentGUIElement
     * @return
     */
    private boolean isRelevant(IGUIElement currentGUIElement) {
        if (currentGUIElement == null) {
            return false;
        }
        
        if ((currentGUIElement instanceof IButton) ||
            (currentGUIElement instanceof ICheckBox) ||
            (currentGUIElement instanceof IComboBox) ||
            (currentGUIElement instanceof IListBox) ||
            (currentGUIElement instanceof IMenuButton) ||
            (currentGUIElement instanceof ITextArea) ||
            (currentGUIElement instanceof ITextField))
        {
            return true;
        }
        
        return false;
    }
}
