//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIView;

/**
 * <p>
 * This is a GUI element representing tags in HTML documents. Each tag belongs to a certain
 * document. Each page element has a tag name and either an id or at least
 * an index in the list of siblings of the same type.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLPageElement extends HTMLGUIElement {

    /**
     * <p>
     * default serial version UID
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * instantiates a page element with it parent (either a document or another page element)
     * </p>
     *
     * @param specification the specification of the page element
     * @param parent        the parent GUI element
     */
    public HTMLPageElement(HTMLPageElementSpec specification, HTMLGUIElement parent) {
        super(specification, parent);
    }

    /**
     * <p>
     * returns the name of the tag represented by this specification
     * </p>
     *
     * @return the name of the tag represented by this specification
     */
    public String getTagName() {
        return ((HTMLPageElementSpec) super.getSpecification()).getTagName();
    }

    /**
     * <p>
     * returns the id of the tag represented by this specification
     * </p>
     *
     * @return the id of the tag represented by this specification
     */
    public String getHtmlId() {
        return ((HTMLPageElementSpec) super.getSpecification()).getHtmlId();
    }

    /**
     * <p>
     * returns the index of the tag
     * </p>
     *
     * @return the index of the tag
     */
    public int getIndex() {
        return ((HTMLPageElementSpec) super.getSpecification()).getIndex();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getView()
     */
    @Override
    public IGUIView getView() {
        return ((HTMLPageElementSpec) super.getSpecification()).getPage();
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getDistanceTo(IGUIElement)
     */
    @Override
    public double getDistanceTo(IGUIElement otherElement) {
        if ((otherElement instanceof HTMLServer) || (otherElement instanceof HTMLDocument)) {
            // use the implementation provided by the server or document
            return otherElement.getDistanceTo(this);
        }
        else if (otherElement instanceof HTMLPageElement) {
            if (equals(otherElement)) {
                return DISTANCE_NONE;
            }
            
            HTMLDocumentSpec documentSpec1 =
                ((HTMLPageElementSpec) this.getSpecification()).getPage();
            HTMLDocumentSpec documentSpec2 =
                ((HTMLPageElementSpec) otherElement.getSpecification()).getPage();
            
            if (!documentSpec1.getSimilarity(documentSpec2)) {
                if (!documentSpec1.getServer().getSimilarity(documentSpec2.getServer())) {
                    return DISTANCE_DISTINCT_SERVER;
                }
                else {
                    return DISTANCE_SAME_SERVER;
                }
            }
            else {
                // check if they have the same parent div element. If so, they are very close.
                // If not, they may be structured completely differently
                IGUIElement parentDiv1 = this;
                
                while (parentDiv1 != null) {
                    if ((parentDiv1 instanceof HTMLPageElement) &&
                        ("div".equals(((HTMLPageElement) parentDiv1).getTagName())))
                    {
                        break;
                    }
                    else {
                        parentDiv1 = parentDiv1.getParent();
                    }
                }
                
                IGUIElement parentDiv2 = (HTMLPageElement) otherElement;
                
                while (parentDiv2 != null) {
                    if ((parentDiv2 instanceof HTMLPageElement) &&
                        ("div".equals(((HTMLPageElement) parentDiv2).getTagName())))
                    {
                        break;
                    }
                    else {
                        parentDiv2 = parentDiv2.getParent();
                    }
                }
                
                // a check for the identity of the objects is sufficient. That they resist on the
                // same document was checked beforehand. So even a condense of the GUI model should
                // not cause an invalid result here.
                if ((parentDiv1 == parentDiv2) ||
                    ((parentDiv1 != null) && (parentDiv1.equals(parentDiv2))))
                {
                    return DISTANCE_SAME_DIV;
                }
                else {
                    return DISTANCE_SAME_DOCUMENT;
                }
            }
        }
        
        return DISTANCE_DISTINCT_SERVER;
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.plugin.html.guimodel.HTMLGUIElement#getElementDescriptor()
     */
    @Override
    protected String getElementDescriptor() {
        return getTagName();
    }

}
