//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.http.eventcore;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import org.apache.commons.collections4.set.ListOrderedSet;

/**
 * <p>
 * Handles all request bodies of equal SOAP events. Can be used to view either all request bodies sent to an operation or to randomly draw one of the bodies.
 * </p>
 * 
 * @author Steffen Herbold
 */
public class EqualSOAPDataMap implements Serializable {
    
    /**  */
    private static final long serialVersionUID = 1L;

    /**
     * Singleton. Handle to only instance of this class
     */
    transient private static EqualSOAPDataMap INSTANCE = new EqualSOAPDataMap();
    
    /**
     * Map with all soapRequestBodies for all equal {@link SimpleSOAPEventType}s
     */
    private Map<SimpleSOAPEventType, ListOrderedSet<String>> soapRequestBodies = new HashMap<>();
    
    /**
     * random number generator for picking a random request body
     */
    transient private final Random random;
    
    /**
     * <p>
     * return the instance of the class
     * </p>
     *
     * @return
     */
    public static EqualSOAPDataMap getInstance() {
        return INSTANCE;
    }
    
    private EqualSOAPDataMap() {
        // private constructor to avoid initialization
        random = new Random();
    }
    
    /**
     * <p>
     * Manual serialization of the object. Necessary to guarantee the singleton
     * property.
     * </p>
     * 
     * @param s
     *            output stream for the serialization
     * @throws IOException
     *             thrown if there is problem writing to the output stream
     */
    private void writeObject(ObjectOutputStream s) throws IOException {
            s.defaultWriteObject();
            s.writeObject(soapRequestBodies);
    }

    /**
     * <p>
     * Manual de-serialization of the object. Necessary to guarantee the
     * singleton property.
     * 
     * @param s
     *            input stream for the de-serialization
     * @throws IOException
     *             thrown if there is problem reading from the input stream
     * @throws ClassNotFoundException
     *             thrown if there is a problem reading from the input stream
     */
    @SuppressWarnings("unchecked")
    private void readObject(ObjectInputStream s) throws IOException,
                    ClassNotFoundException {
            s.defaultReadObject();
            if (INSTANCE == null) {
                INSTANCE = new EqualSOAPDataMap();
            }
            INSTANCE.soapRequestBodies = (Map<SimpleSOAPEventType, ListOrderedSet<String>>) s.readObject();
    }

    /**
     * <p>
     * Manual de-serialization to guarantee the singleton property.
     * </p>
     * 
     * @return instance of the container
     */
    private Object readResolve() {
            return INSTANCE;
    }

    /**
     * <p>
     * Adds a new body to the map.
     * </p>
     *
     * @param simpleSOAPEventType
     */
    public void add(SimpleSOAPEventType simpleSOAPEventType, String soapRequestBody) {
        if( soapRequestBody!=null ) {
            ListOrderedSet<String> requestBodySet = soapRequestBodies.get(simpleSOAPEventType);
            if( requestBodySet==null ) {
                requestBodySet = new ListOrderedSet<>();
                soapRequestBodies.put(simpleSOAPEventType, requestBodySet);
            }
            requestBodySet.add(soapRequestBody);
        }
    }
    
    /**
     * <p>
     * Retrieves all bodies associated with the simpleSoapEventType; null if not found
     * </p>
     *
     * @param simpleSoapEventType
     * @return
     */
    public Set<String> getAll(SimpleSOAPEventType simpleSoapEventType) {
        return Collections.unmodifiableSet(soapRequestBodies.get(simpleSoapEventType));
    }
    
    /**
     * <p>
     * Randomly draws one of the SOAP event type bodies associated with the event
     * </p>
     *
     * @param simpleSOAPEventType
     * @return
     */
    public String getRandom(SimpleSOAPEventType simpleSOAPEventType) {
        ListOrderedSet<String> requestBodySet = soapRequestBodies.get(simpleSOAPEventType);
        if( requestBodySet==null || requestBodySet.isEmpty() ) {
            throw new RuntimeException("no request body known for SimpleSOAPEventType: " + simpleSOAPEventType);
        }
        
        return requestBodySet.get(random.nextInt(requestBodySet.size()));
    }
    
    /**
     * <p>
     * resets the internal map by creating new one
     * </p>
     *
     */
    public void reset() {
        soapRequestBodies = new HashMap<>();
    }
}