//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.guimodel;

import de.ugoe.cs.autoquest.eventcore.guimodel.AbstractDefaultGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIView;

/**
 * <p>
 * Base class for all JFC GUI elements.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public class JFCGUIElement extends AbstractDefaultGUIElement {

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Specification of the GUI Element
     * </p>
     */
    private JFCGUIElementSpec specification;

    /**
     * <p>
     * Constructor. Creates a new JFCGUIElement.
     * </p>
     * 
     * @param specification
     *            specification of created GUI element
     * @param parent
     *            parent of the created GUI element; null means that the element is a top-level
     *            window
     */
    public JFCGUIElement(JFCGUIElementSpec specification, JFCGUIElement parent) {
        super(specification, parent);
        this.specification = specification;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.IEventTarget#getPlatform()
     */
    @Override
    public String getPlatform() {
        return "JFC";
    }

    /**
     * <p>
     * Returns the type of the GUI element, i.e., the name of its Java class.
     * </p>
     * 
     * @return the Java class name
     */
    public String getJavaType() {
        return specification.getType();
    }

    /**
     * <p>
     * Returns the name of the GUI element.
     * </p>
     * 
     * @return the name
     */
    public String getName() {
        return specification.getName();
    }

    /**
     * <p>
     * Returns the icon of the GUI element.
     * </p>
     * 
     * @return the icon
     */
    String getIcon() {
        return specification.getIcon();
    }

    /**
     * <p>
     * Returns the index of the GUI element.
     * </p>
     * 
     * @return the index
     */
    int getIndex() {
        return specification.getIndex();
    }

    /**
     * <p>
     * Returns the object hash of the GUI element.
     * </p>
     * 
     * @return the object hash
     */
    int getElementHash() {
        return specification.getElementHash();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#updateSpecification(de.ugoe.cs.autoquest
     * .eventcore .guimodel.IGUIElementSpec)
     */
    @Override
    public void updateSpecification(IGUIElementSpec updateSpecification) {
        if (updateSpecification instanceof JFCGUIElementSpec) {
            specification.update(((JFCGUIElementSpec) updateSpecification));
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String getStringIdentifier() {
        String str = this.toString();
        if (getParent() != null) {
            return str + "<-" + getParent().getStringIdentifier();
        }
        return str;
    }

    public String getJacaretoHierarchy() {
        String str;

        // get the Java classname, ignore the package hierarchy if present
        String[] parts = getSpecification().getType().split("\\.");

        // find the correct Jacareto index
        // jacareto indices start at 1
        int jacIndex = ((JFCGUIElementSpec) getSpecification()).getAltIndex() + 1;
        str = parts[parts.length - 1] + "_(" + jacIndex + ")";

        if (getParent() != null) {
            return ((JFCGUIElement) getParent()).getJacaretoHierarchy() + "." + str;
        }
        return str;
    }

    public String getJacaretoRoot() {
        return getJacaretoHierarchy().split("\\.")[0];
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String str =
            getElementDescriptor() + "(" + getName() + "," + getElementHash() + "," + getIcon() +
                "," + getIndex() + ")";
        return str;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getView()
     */
    @Override
    public IGUIView getView() {
        IGUIElement element = this;
        
        while ((element != null) && (!(element instanceof IGUIView))) {
            if (!(element.getParent() instanceof JFCTabbedPane)) {
                element = element.getParent();
            }
            else {
                // break, as all children of a tabbed pane are always views
                break;
            }
        }
        
        return (IGUIView) element;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement#getDistanceTo(IGUIElement)
     */
    @Override
    public double getDistanceTo(IGUIElement otherElement) {
        throw new UnsupportedOperationException("not implemented yet");
    }

    /**
     * <p>
     * A short string describing the GUI element, e.g., Button, Canvas, or ScrollBar.
     * </p>
     * 
     * @return short element descriptor
     */
    protected String getElementDescriptor() {
        return "Default";
    }

}
