//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.http;

import static org.junit.Assert.*;

import java.io.File;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;

import org.junit.BeforeClass;
import org.junit.Test;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.EventUtils;
import de.ugoe.cs.autoquest.plugin.http.HTTPLogParser;
import de.ugoe.cs.autoquest.plugin.http.SOAPUtils;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SimpleSOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.eventcore.SimpleSOAPEventType.CallType;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.TextConsole;

/**
 * <p>
 * Tests for the SOAPUtils
 * </p>
 * 
 * @author Steffen Herbold
 */
public class SOAPUtilsTest {
    
    @BeforeClass
    public static void setupBeforeClass() {
        new TextConsole(Level.FINEST);
    }

    @Test
    public void testConvertToSimpleSOAPEvent_1() throws Exception {
        HTTPLogParser parser =
            new HTTPLogParser(new File(ClassLoader
                .getSystemResource("testParseFile_3_properties.txt").getFile()));
        parser.parseFile(new File(ClassLoader.getSystemResource("testParseFile_3_logfile.log")
            .getFile()));
        Collection<List<Event>> sequences = parser.getSequences();

        assertNotNull(sequences);
        assertFalse(sequences.isEmpty());

        Collection<List<Event>> newSequences = SOAPUtils.convertToSimpleSOAPEvent(sequences, false);
        
        assertNotNull(newSequences);
        assertEquals(sequences.size(), newSequences.size());

        // compare sequences
        Iterator<List<Event>> oldIter = sequences.iterator();
        Iterator<List<Event>> newIter = newSequences.iterator();
        
        while( oldIter.hasNext() ) {
            Iterator<Event> oldSeqIter = oldIter.next().iterator();
            Iterator<Event> newSeqIter = newIter.next().iterator();
    
            while (oldSeqIter.hasNext()) {
                Event oldEvent = oldSeqIter.next();
                Event newEvent = newSeqIter.next();
    
                if (oldEvent.getType() instanceof SOAPEventType) {
                    assertTrue(newEvent.getType() instanceof SimpleSOAPEventType);
                    SOAPEventType oldEventType = (SOAPEventType) oldEvent.getType();
                    SimpleSOAPEventType newEventType = (SimpleSOAPEventType) newEvent.getType();
                    assertEquals(oldEventType.getCalledMethod(), newEventType.getCalledMethod());
                    assertEquals(oldEventType.getServiceName(), newEventType.getServiceName());
                    assertEquals(oldEventType.getClientName(), newEventType.getClientName());
                    assertNull(newEvent.getTarget());
                }
                else {
                    assertSame(oldEvent, newEvent);
                }
            }
        }
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_1() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        
        List<Event> expectedSequence = new LinkedList<>(sequence);
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName());
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_2() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        
        List<Event> expectedSequence = new LinkedList<Event>();
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName());
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_3() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        
        List<Event> expectedSequence = new LinkedList<Event>();
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName());
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_4() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(Event.STARTEVENT);
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(Event.ENDEVENT);
        
        List<Event> expectedSequence = new LinkedList<>(sequence);
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName());
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_5() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(Event.STARTEVENT);
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(Event.ENDEVENT);
        
        List<Event> expectedSequence = new LinkedList<Event>();
        expectedSequence.add(Event.STARTEVENT);
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        expectedSequence.add(Event.ENDEVENT);
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName() );
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }
    
    @Test
    public void testDropInvalidResponseRequestPairs_6() throws Exception {
        List<Event> sequence = new LinkedList<Event>();
        sequence.add(Event.STARTEVENT);
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.REQUEST)));
        sequence.add(new Event(new SimpleSOAPEventType("op2", "foo", "bar", null, null, CallType.RESPONSE)));
        sequence.add(Event.ENDEVENT);
        
        List<Event> expectedSequence = new LinkedList<Event>();
        expectedSequence.add(Event.STARTEVENT);
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.REQUEST)));
        expectedSequence.add(new Event(new SimpleSOAPEventType("op1", "foo", "bar", null, null, CallType.RESPONSE)));
        expectedSequence.add(Event.ENDEVENT);
        
        List<Event> originalSequence = new LinkedList<>(sequence);
        
        List<Event> validSequence = SOAPUtils.dropInvalidResponseRequestPairs(sequence);
        Console.traceln(Level.INFO, getCurrentMethodName());
        EventUtils.traceSequence(Level.INFO, validSequence);
        
        assertEquals(expectedSequence, validSequence);
        assertEquals(originalSequence, sequence);
    }

    private  String getCurrentMethodName() {
         StackTraceElement stackTraceElements[] = (new Throwable()).getStackTrace();
         return stackTraceElements[1].toString();
    }
    
}
