//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usageprofiles;

import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Implements a round-trip buffered memory of a specified length that can be used to remember the
 * recent history. Every event that happend longer ago than the length of the memory is forgotten,
 * hence the memory is incomplete.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 * 
 * @param <T>
 *            Type which is memorized.
 */
public class IncompleteMemory<T> implements IMemory<T> {

    /**
     * <p>
     * Maximum length of the memory.
     * </p>
     */
    private int length;

    /**
     * <p>
     * Internal storage of the history.
     * </p>
     */
    private List<T> history;

    /**
     * <p>
     * Constructor. Creates a new IncompleteMemory.
     * </p>
     * 
     * @param length
     *            number of recent events that are remembered
     * @throws IllegalArgumentException
     *             This exception is thrown if the length is smaller than 1
     */
    public IncompleteMemory(int length) {
        if (length < 1) {
            throw new IllegalArgumentException("Length of IncompleteMemory must be at least 1.");
        }
        this.length = length;
        history = new LinkedList<T>();
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usageprofiles.IMemory#add(java.lang.Object)
     */
    @Override
    public void add(T state) {
        if (history.size() == length) {
            history.remove(0);
        }
        history.add(state);
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.autoquest.usageprofiles.IMemory#getLast(int)
     */
    @Override
    public List<T> getLast(int num) {
        if (num < 1) {
            return new LinkedList<T>();
        }
        else {
            return new LinkedList<T>(history.subList(Math.max(0, history.size() - num),
                                                     history.size())); // defensive copy
        }
    }

    /**
     * <p>
     * Returns the current length of the memory. This can be less than {@link #length}, if the
     * overall history is less than {@link #length}.
     * </p>
     * 
     * @return length of the current memory
     */
    public int getLength() {
        return history.size();
    }
}
