//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.sequences;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to split all sequences of a collection of sequences into subsequences depending on
 * the timestamps of the events. If the difference of the timestamps of two subsequent events is
 * larger than the provided number of milliseconds, the sequence to which the events belong is
 * split up into two subsequences between the two events. 
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDsplitSequences implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        long diff;
        String sequencesName;
        String newSequencesName;
        try {
            diff = Long.parseLong((String) parameters.get(0));
            sequencesName = (String) parameters.get(1);
            if (parameters.size() > 2) {
                newSequencesName = (String) parameters.get(2);
            }
            else {
                newSequencesName = sequencesName;
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("must provide an integer for the maximum " +
                                               "timestamp difference and a sequences name");
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;
        
        Collection<List<Event>> newSequences = new LinkedList<List<Event>>();
        
        Console.println("splitting " + sequences.size() + " sequences");
        for (List<Event> sequence : sequences) {
            Event previous = null;
            List<Event> resultingSequence = new LinkedList<Event>();
            for (Event currentEvent : sequence) {
                if (currentEvent.getTimestamp() < 0) {
                    Console.printerrln("at least one of the events does not have a correct " +
                                       "timestamp required for the merging of sequences and " +
                                       "sorting of events");
                }
                
                if ((previous != null) &&
                    (Math.abs(previous.getTimestamp() - currentEvent.getTimestamp()) > diff))
                {
                    newSequences.add(resultingSequence);
                    resultingSequence = new LinkedList<Event>();
                }
                
                resultingSequence.add(currentEvent);
                previous = currentEvent;
            }
            
            newSequences.add(resultingSequence);
        }
        
        if (GlobalDataContainer.getInstance().addData(newSequencesName, newSequences)) {
            CommandHelpers.dataOverwritten(newSequencesName);
        }
        
        Console.println("sequences splitted");
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "splitSequences <maximumTimestampDifference> <sequencesName> {<newSequences>}";
    }

}
