//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskInstanceTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISelection;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IStructuringTemporalRelationship;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInfo;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstanceVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IUserSession;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class TaskTreeTestRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        checkTimestamps(taskModel);
        checkTaskInfos(taskModel);
        checkEventTaskInstances(taskModel);
        checkTraversal(taskModel);

        return results;
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private void checkEventTaskInstances(ITaskModel taskModel) {
        Set<IEventTaskInstance> instancesInModel = new HashSet<>();
        
        for (ITask task : taskModel.getTasks()) {
            if (task instanceof IEventTask) {
                for (ITaskInstance instance : task.getInstances()) {
                    instancesInModel.add((IEventTaskInstance) instance);
                }
            }
        }
        
        final Set<IEventTaskInstance> instancesInSessions = new HashSet<>();
        
        for (IUserSession session : taskModel.getUserSessions()) {
            for (ITaskInstance rootInstance : session) {
                rootInstance.accept(new DefaultTaskInstanceTraversingVisitor() {
                    @Override
                    public void visit(IEventTaskInstance eventTaskInstance) {
                        instancesInSessions.add(eventTaskInstance);
                    }
                });
            }
        }
        
        if (instancesInModel.size() != instancesInSessions.size()) {
            throw new RuntimeException("instances not equal");
        }
        else {
            for (IEventTaskInstance instance : instancesInModel) {
                if (!instancesInSessions.contains(instance)) {
                    throw new RuntimeException("instance of model not contained in sessions");
                }
            }
            
            for (IEventTaskInstance instance : instancesInSessions) {
                if (!instancesInModel.contains(instance)) {
                    throw new RuntimeException("instance of sessions not contained in model");
                }
            }
        }
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private void checkTraversal(ITaskModel taskModel) {
        for (ITask task : taskModel.getTasks()) {
            final List<ITask> traversal1 = new ArrayList<>();
            
            task.accept(new DefaultTaskTraversingVisitor() {
                @Override
                public void visit(IEventTask eventTask) {
                    traversal1.add(eventTask);
                    super.visit(eventTask);
                }

                @Override
                public void visit(IStructuringTemporalRelationship relationship) {
                    traversal1.add(relationship);
                    super.visit(relationship);
                }
                
            });
            
            final List<ITask> traversal2 = new ArrayList<>();
            
            task.accept(new DefaultTaskTraversingVisitor() {
                @Override
                public void visit(IEventTask eventTask) {
                    traversal2.add(eventTask);
                    super.visit(eventTask);
                }

                @Override
                public void visit(ISelection selection) {
                    traversal2.add(selection);
                    super.visit(selection);
                }

                @Override
                public void visit(ISequence sequence) {
                    traversal2.add(sequence);
                    super.visit(sequence);
                }
            });
            
            if (traversal1.size() != traversal2.size()) {
                throw new RuntimeException("traversals not equal");
            }
            else {
                for (int i = 0; i < traversal1.size(); i++) {
                    if (!traversal1.get(i).equals(traversal2.get(i))) {
                        throw new RuntimeException("traversals not equal at position " + i);
                    }
                }
            }
        }
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private void checkTimestamps(ITaskModel taskModel) {
        for (IUserSession session : taskModel.getUserSessions()) {
            ITaskInstanceVisitor visitor = new DefaultTaskInstanceTraversingVisitor() {
                Event lastEvent = null;
                
                @Override
                public void visit(IEventTaskInstance eventTaskInstance) {
                    if (lastEvent != null) {
                        if (eventTaskInstance.getEvent().getTimestamp() < lastEvent.getTimestamp()) {
                            System.out.println("timestamp problem encountered:");
                            System.out.println("  " + lastEvent.getTimestamp() + "  " + lastEvent);
                            System.out.println("  " + eventTaskInstance.getEvent().getTimestamp() + "  " + eventTaskInstance.getEvent());
                        }
                    }
                    
                    lastEvent = eventTaskInstance.getEvent();
                }
                
            };
            
            for (ITaskInstance instance : session) {
                visitor.visit(instance);
            }
        }
        
    }

    /**
     * <p>
     * TODO: comment
     * </p>
     *
     * @param taskModel
     */
    private void checkTaskInfos(ITaskModel taskModel) {
        for (ITask task : taskModel.getTasks()) {
            ITaskInfo taskInfo = taskModel.getTaskInfo(task);
            if (taskInfo == null) {
                System.out.println("task " + task + " has no associated task infos");
            }
        }
    }
}
