//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.ui.swt;

import java.util.Collection;

import org.eclipse.swt.widgets.Dialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.layout.GridData;

import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.util.console.GlobalDataContainer;

import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;

public class SequencesDialog extends Dialog {

    private String sequencesName;

    private List sequenceList;
    private Collection<java.util.List<Event>> sequences;
    private GUIModel guiModel;

    protected Shell shell;

    /**
     * Create the dialog.
     * 
     * @param parent
     * @param style
     */
    public SequencesDialog(Shell parent, int style) {
        super(parent, style);
        setText("SWT Dialog");
    }

    /**
     * Open the dialog.
     */
    public void open(String sequencesName) {
        this.sequencesName = sequencesName;
        sequences = null;
        createContents();
        shell.open();
        shell.layout();
        Display display = getParent().getDisplay();
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch()) {
                display.sleep();
            }
        }
    }

    /**
     * Create contents of the dialog.
     */
    private void createContents() {
        shell = new Shell(getParent(), SWT.SHELL_TRIM | SWT.BORDER | SWT.APPLICATION_MODAL);
        shell.setSize(248, 299);
        shell.setText(getText());
        shell.setLayout(new GridLayout(2, false));

        sequenceList = new List(shell, SWT.BORDER | SWT.V_SCROLL);
        sequenceList.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 2, 1));
        updateSequenceList();

        Button btnShow = new Button(shell, SWT.NONE);
        btnShow.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                int index = sequenceList.getSelectionIndex();
                if (index == -1) {
                    MessageBox messageBox = new MessageBox(shell, SWT.ERROR);
                    messageBox.setMessage("No sequence selected!");
                    messageBox.setText("Error");
                    messageBox.open();
                }
                else {
                    EditSequenceDialog editSequenceDialog =
                        new EditSequenceDialog(shell, SWT.NONE, guiModel);
                    int counter = 0;
                    java.util.List<Event> selectedSequence = null;
                    for (java.util.List<Event> sequence : sequences) {
                        if (counter == index) {
                            selectedSequence = sequence;
                            break;
                        }
                        counter++;
                    }
                    editSequenceDialog.open(selectedSequence);
                    updateSequenceList();
                }
            }
        });
        btnShow.setText("Show");

        Button btnClose = new Button(shell, SWT.NONE);
        btnClose.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                shell.dispose();
            }
        });
        btnClose.setText("Close");

    }

    @SuppressWarnings("unchecked")
    private void updateSequenceList() {
        sequenceList.removeAll();
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            sequences = (Collection<java.util.List<Event>>) dataObject;
            int seqDigits = Integer.toString(sequences.size()).length();
            int counter = 1;
            for (java.util.List<Event> sequence : sequences) {
                String seqName =
                    "#" + String.format("%0" + seqDigits + "d", counter) + ": " + sequence.size();
                sequenceList.add(seqName);
                counter++;
            }
            Object targetObject =
                GlobalDataContainer.getInstance().getData(sequencesName + "_targets");
            guiModel = null;
            if (targetObject instanceof GUIModel) {
                guiModel = (GUIModel) targetObject;
            }
        }
        else {
            MessageBox messageBox = new MessageBox(shell, SWT.ERROR);
            messageBox.setMessage("Internal error. Sequences object not of expected type!");
            messageBox.setText("Error");
            messageBox.open();
        }
    }

}
