//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIterationInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

// TODO: Auto-generated Javadoc
/**
 * <p>
 * This class is capable of comparing any task which is not an iteration with an
 * iteration. This is needed, because iterations may iterate exactly that task.
 * In this case, the iteration would be equal to that task if it was executed
 * exactly once. The rule returns lexically equal, it the child of the iteration
 * is lexically equal to the task or if the child of the iteration is a
 * selection and this selections contains a lexically equal task. The same
 * applies for syntactical and semantical equality.
 * </p>
 * 
 * @author Patrick Harms
 */
public class TaskAndIterationComparisonRule implements TaskComparisonRule {

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areLexicallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areLexicallyEqual(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean areLexicallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.LEXICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSemanticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSemanticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSemanticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SEMANTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITask, ITask)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
		final TaskEquality equality = getEquality(task1, task2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#areSyntacticallyEqual(ITaskInstance,
	 * ITaskInstance)
	 */
	@Override
	public boolean areSyntacticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2) {
		final TaskEquality equality = getEquality(instance1, instance2,
				TaskEquality.SYNTACTICALLY_EQUAL);
		return (equality != null)
				&& (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param child1
	 *            the first task to be compared
	 * @param child2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITask child1, ITask child2,
			TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance()
					.compare(child1, child2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				child1, child2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/**
	 * <p>
	 * used to to call the task equality rule manager for the comparison of the
	 * two provided children. If no required equality level is provided, than
	 * the most concrete equality is returned. Otherwise, the required equality
	 * is returned as long as the children are equal on that level.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality
	 */
	private TaskEquality callRuleManager(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		if (requiredEqualityLevel == null) {
			return TaskEqualityRuleManager.getInstance().compare(taskInstance1,
					taskInstance2);
		} else if (TaskEqualityRuleManager.getInstance().areAtLeastEqual(
				taskInstance1, taskInstance2, requiredEqualityLevel)) {
			return requiredEqualityLevel;
		} else {
			return TaskEquality.UNEQUAL;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITask, ITask)
	 */
	@Override
	public TaskEquality compare(ITask task1, ITask task2) {
		return getEquality(task1, task2, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#compare(ITaskInstance, ITaskInstance)
	 */
	@Override
	public TaskEquality compare(ITaskInstance instance1, ITaskInstance instance2) {
		return getEquality(instance1, instance2, null);
	}

	/**
	 * <p>
	 * compares two tasks with each other checking for the provided required
	 * level of equality. One of the tasks must be an iteration, the other one
	 * not. If this is not the case, the method returns null. The returned
	 * equality level is at most lexical equality as the iteration can not be
	 * identical to something not being an iteration.
	 * </p>
	 * 
	 * @param task1
	 *            the first task to be compared
	 * @param task2
	 *            the second task to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITask task1, ITask task2,
			TaskEquality requiredEqualityLevel) {
		IIteration iteration = null;
		ITask task = null;

		if (task1 instanceof IIteration) {
			if (task2 instanceof IIteration) {
				// the rule is not responsible for two iterations
				return null;
			}

			iteration = (IIteration) task1;
			task = task2;
		} else if (task2 instanceof IIteration) {
			if (task1 instanceof IIteration) {
				// the rule is not responsible for two iterations
				return null;
			}

			iteration = (IIteration) task2;
			task = task1;
		} else {
			return null;
		}

		final ITask child = iteration.getMarkedTask();

		// now, that we found the iteration and the task, lets compare the child
		// of the iteration
		// with the task.
		if (child == null) {
			return null;
		}

		final TaskEquality taskEquality = callRuleManager(child, task,
				requiredEqualityLevel);

		// although the subtask may be identical to the task, we can not return
		// identical, as
		// the iteration is not identical to the task, but at most lexically
		// equal
		if (taskEquality == TaskEquality.IDENTICAL) {
			return TaskEquality.LEXICALLY_EQUAL;
		} else {
			return taskEquality;
		}

	}

	/**
	 * <p>
	 * compares two task instances with each other checking for the provided
	 * required level of equality. One of the task instances must be a
	 * iteration, the other one not. If this is not the case, the method returns
	 * null. The returned equality level is at most lexical equality as the
	 * iteration can not be identical to something not being a iteration.
	 * </p>
	 * 
	 * @param taskInstance1
	 *            the first task instance to be compared
	 * @param taskInstance2
	 *            the second task instance to be compared
	 * @param requiredEqualityLevel
	 *            the equality level to be checked for
	 * 
	 * @return the determined equality.
	 */
	private TaskEquality getEquality(ITaskInstance taskInstance1,
			ITaskInstance taskInstance2, TaskEquality requiredEqualityLevel) {
		IIterationInstance iteration = null;
		ITaskInstance task = null;

		if (taskInstance1 instanceof IIterationInstance) {
			if (taskInstance2 instanceof IIterationInstance) {
				// the rule is not responsible for two iterations
				return null;
			}

			iteration = (IIterationInstance) taskInstance1;
			task = taskInstance2;
		} else if (taskInstance2 instanceof IIterationInstance) {
			if (taskInstance1 instanceof IIterationInstance) {
				// the rule is not responsible for two iterations
				return null;
			}

			iteration = (IIterationInstance) taskInstance2;
			task = taskInstance1;
		} else {
			return null;
		}

		// now, that we found the iteration and the task, lets compare the
		// children of the iteration
		// with the task.

		if (iteration.size() < 1) {
			return null;
		}

		TaskEquality mostConcreteNodeEquality = null;

		for (final ITaskInstance child : iteration) {
			final TaskEquality taskEquality = callRuleManager(child, task,
					requiredEqualityLevel);

			if (taskEquality != TaskEquality.UNEQUAL) {
				if (mostConcreteNodeEquality == null) {
					mostConcreteNodeEquality = taskEquality;
				} else if (mostConcreteNodeEquality.isAtLeast(taskEquality)) {
					mostConcreteNodeEquality = taskEquality;

				}

				if ((requiredEqualityLevel != null)
						&& (mostConcreteNodeEquality
								.isAtLeast(requiredEqualityLevel))) {
					// if we found one child of the selection that is as equal
					// as required, then
					// we can consider the selection to be sufficiently equal to
					// the other task.
					// So we break up checking further children.
					break;
				}
			}
		}

		// although the subtask may be identical to the task, we can not return
		// identical, as
		// the selection is not identical to the task, but at most lexically
		// equal
		if (mostConcreteNodeEquality == TaskEquality.IDENTICAL) {
			return TaskEquality.LEXICALLY_EQUAL;
		} else {
			return mostConcreteNodeEquality;
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITask, ITask)
	 */
	@Override
	public boolean isApplicable(ITask task1, ITask task2) {
		return ((task1 instanceof IIteration) && (!(task2 instanceof IIteration)))
				|| ((task2 instanceof IIteration) && (!(task1 instanceof IIteration)));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see TaskComparisonRule#isApplicable(ITaskInstance, ITaskInstance)
	 */
	@Override
	public boolean isApplicable(ITaskInstance instance1, ITaskInstance instance2) {
		return isApplicable(instance1.getTask(), instance2.getTask());
	}
}
