//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

// TODO: Auto-generated Javadoc
/**
 * <p>
 * A task comparison rule is used by the {@link TaskEqualityRuleManager} to
 * compare tasks and task instances with each other. It provides several methods
 * to be called for a comparison.
 * </p>
 *
 * @author 2012, last modified by $Author: patrick$
 * @version $Revision: $ $Date: 19.02.2012$
 */
public interface TaskComparisonRule {

	/**
	 * <p>
	 * checks, if the provided tasks are lexically equal
	 * </p>.
	 *
	 * @param task1            the first task to compare
	 * @param task2            the second task to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areLexicallyEqual(ITask task1, ITask task2);

	/**
	 * <p>
	 * checks, if the provided task instances are lexically equal
	 * </p>.
	 *
	 * @param instance1            the first task instance to compare
	 * @param instance2            the second task instance to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areLexicallyEqual(ITaskInstance instance1,
			ITaskInstance instance2);

	/**
	 * <p>
	 * checks, if the provided tasks are semantically equal
	 * </p>.
	 *
	 * @param task1            the first task to compare
	 * @param task2            the second task to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areSemanticallyEqual(ITask task1, ITask task2);

	/**
	 * <p>
	 * checks, if the provided task instances are semantically equal
	 * </p>.
	 *
	 * @param instance1            the first task instance to compare
	 * @param instance2            the second task instance to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areSemanticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2);

	/**
	 * <p>
	 * checks, if the provided tasks are syntactically equal
	 * </p>.
	 *
	 * @param task1            the first task to compare
	 * @param task2            the second task to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areSyntacticallyEqual(ITask task1, ITask task2);

	/**
	 * <p>
	 * checks, if the provided task instances are syntactically equal
	 * </p>.
	 *
	 * @param instance1            the first task instance to compare
	 * @param instance2            the second task instance to compare
	 * @return true, if the tasks are equal, false else
	 */
	public boolean areSyntacticallyEqual(ITaskInstance instance1,
			ITaskInstance instance2);

	/**
	 * <p>
	 * compares two tasks with each other. The result of the method is either a
	 * task equality or null. If it is null, it means, that the rule is not able
	 * to correctly compare the two given tasks
	 * </p>
	 * 
	 * @param task1
	 *            the first task to compare
	 * @param task2
	 *            the second task to compare
	 * 
	 * @return as described
	 */
	public TaskEquality compare(ITask task1, ITask task2);

	/**
	 * <p>
	 * compares two task instances with each other. The result of the method is
	 * either a task instance equality or null. If it is null, it means, that
	 * the rule is not able to correctly compare the two given task instances
	 * </p>
	 * 
	 * @param instance1
	 *            the first task instance to compare
	 * @param instance2
	 *            the second task instance to compare
	 * 
	 * @return as described
	 */
	public TaskEquality compare(ITaskInstance instance1, ITaskInstance instance2);

	/**
	 * <p>
	 * checks if the rule is applicable for comparing the two provided tasks
	 * </p>.
	 *
	 * @param task1            the first task to compare
	 * @param task2            the second task to compare
	 * @return true, if the rule is applicable, false else
	 */
	public boolean isApplicable(ITask task1, ITask task2);

	/**
	 * <p>
	 * checks if the rule is applicable for comparing the two provided task
	 * instances
	 * </p>.
	 *
	 * @param instance1            the first task instance to compare
	 * @param instance2            the second task instance to compare
	 * @return true, if the rule is applicable, false else
	 */
	public boolean isApplicable(ITaskInstance instance1, ITaskInstance instance2);

}
