//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.commands;

import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.autoquest.plugin.jfc.JFCSimplifiedLogParser;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElement;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCGUIElementSpec;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCMenu;
import de.ugoe.cs.autoquest.plugin.jfc.guimodel.JFCMenuButton;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to generate sessions in which the JFCGUIElements contain a special index that is required
 * to replay the sequences with Jacareto.
 * </p>
 * 
 * @author Daniel May
 * @version 1.0
 */
public class CMDparseJFCwithJacaretoIndices implements Command {

    private List<String> menuList;

    @Override
    public void run(List<Object> parameters) {
        String filename;
        String sequencesName = "sequences";
        try {
            filename = (String) parameters.get(0);
            if (parameters.size() >= 2) {
                menuList =
                    Files.readAllLines(Paths.get((String) parameters.get(1)),
                                       Charset.defaultCharset());
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException();
        }

        JFCSimplifiedLogParser parser = new JFCSimplifiedLogParser();

        try {
            parser.parseFile(filename);
        }
        catch (Exception e) {
            Console.printerrln("Could not parse " + filename + ": " + e.getMessage());
            return;
        }

        Collection<List<Event>> sequences = parser.getSequences();
        GUIModel targets = parser.getGuiModel();
        targets.condenseModel();

        generateJacaretoIndices(targets.getRootElements(), targets);

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }

        if (GlobalDataContainer.getInstance().addData(sequencesName + "_targets", targets)) {
            CommandHelpers.dataOverwritten(sequencesName + "_targets");
        }
    }

    @Override
    public String help() {
        return "parseJFCwithJacaretoIndices <filename> {<menufile>}";
    }

    private int findPopupMenuIndex(IGUIElement item, GUIModel model) {
        // TODO: refactor
        int index = -1;
        List<IGUIElement> children = model.getChildren(item);
        IGUIElement menuChild = null;

        // try to identify this popup menu by looking at its children
        // find the first menu item child of this item
        for (IGUIElement child : children) {
            if (child instanceof JFCMenuButton || child instanceof JFCMenu) {
                menuChild = child;
                break;
            }
        }

        if (menuChild == null) {
            // this popup menu cannot be identified
            // TODO: exception, logging etc
            return -1;
        }

        // find line that contains this menu item name
        String itemName = ((JFCGUIElement) menuChild).getName().trim().toLowerCase();
        int lineOfItem = -1;

        for (int i = 0; i < menuList.size(); i++) {
            String name = "\"" + menuList.get(i).trim().toLowerCase() + "\"";
            if (name.equals(itemName)) {
                lineOfItem = i;
                break;
            }
        }

        if (lineOfItem == -1) {
            // failed to find this item in the menu file
            // TODO: exception, logging etc
            return -1;
        }

        // find menu item's parent line
        String stripped = menuList.get(lineOfItem).replaceFirst("^ *", "");
        int indent = menuList.get(lineOfItem).length() - stripped.length();

        if (indent != 0) {
            for (int i = lineOfItem; i >= 0; i--) {
                stripped = menuList.get(i).replaceFirst("^ *", "");
                int oldIndent = menuList.get(i).length() - stripped.length();

                if (oldIndent < indent) {
                    lineOfItem = i;
                    break;
                }
            }
        }

        // get the item's indentation
        stripped = menuList.get(lineOfItem).replaceFirst("^ *", "");
        indent = menuList.get(lineOfItem).length() - stripped.length();

        if (indent == 0) {
            // top level menu item, just count in which position it is
            for (int i = 0; i <= lineOfItem; i++) {
                if (!menuList.get(i).startsWith(" ")) {
                    index++;
                }
            }
        }
        else {
            // find the topmenu index in the menu file by going backwards
            for (int i = lineOfItem; i >= 0; i--) {
                stripped = menuList.get(i).replaceFirst("^ *", "");
                int oldIndent = menuList.get(i).length() - stripped.length();

                if (oldIndent < indent) {
                    index = lineOfItem - i;
                    break;
                }
            }
        }

        return index;
    }

    private void generateJacaretoIndices(List<IGUIElement> elements, GUIModel targets) {
        HashMap<String, Integer> typeCount = new HashMap<>();

        for (IGUIElement child : elements) {
            String type = child.getSpecification().getType();
            Integer count = typeCount.get(type);

            if (count == null) {
                count = 0;
                typeCount.put(type, count);
            }
            else {
                typeCount.put(type, ++count);
            }

            if (menuList != null && type.equals("javax.swing.JPopupMenu")) {
                // try to use a workaround for popup menu index problems
                int index = findPopupMenuIndex((IGUIElement) child, targets);
                if (index != -1) {
                    ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(index);
                }
                else {
                    // workaround failed, use normal method as fallback
                    ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(count);
                }
            }
            else {
                ((JFCGUIElementSpec) child.getSpecification()).setAltIndex(count);
            }

            generateJacaretoIndices(targets.getChildren(child), targets);
        }
    }
}
