//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usability;

import java.util.Collection;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.tasktrees.manager.TaskTreeManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * This command generates a task tree based on the provided sequences. It uses the
 * {@link TaskTreeManager} for this purpose. Please consult the documentation of the task tree
 * manager for more details.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDgenerateTaskTree implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "generateTaskTree <sequences> {<tasktree>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String sequencesName;
        String tasktreeName;
        try {
            sequencesName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                tasktreeName = (String) parameters.get(1);
            }
            else {
                tasktreeName = "tasktree";
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("must provide a sequences name");
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;
        
        ITaskModel taskModel = new TaskTreeManager().createTaskModel(sequences);
        
        if (GlobalDataContainer.getInstance().addData(tasktreeName, taskModel)) {
            CommandHelpers.dataOverwritten(tasktreeName);
        }
        
    }

}
