//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.http.eventcore;

import static org.junit.Assert.assertEquals;

import java.math.BigInteger;
import java.util.Properties;

import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPMessage;

import nl.jqno.equalsverifier.EqualsVerifier;
import nl.jqno.equalsverifier.Warning;

import org.junit.Test;

import de.ugoe.cs.autoquest.plugin.http.eventcore.SOAPEventType;
import de.ugoe.cs.autoquest.plugin.http.logdata.Address;
import de.ugoe.cs.autoquest.plugin.http.logdata.HttpExchange;
import de.ugoe.cs.autoquest.plugin.http.logdata.HttpRequest;

/**
 *
 * @author Patrick Harms
 */
public class SOAPEventTypeTest {

    /**
     * 
     */
    @Test
    public void test_EqualsContract_01() {
        EqualsVerifier.forClass(SOAPEventType.class)
            .suppress(Warning.NONFINAL_FIELDS).withRedefinedSuperclass().verify();
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_01() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, null);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_02() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_03() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.sender.sender", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_04() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.sender.sender:1234", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_05() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.server.receiver", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_06() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.server.receiver:5678", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_07() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.port.5678", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_08() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.sender.othersender", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_09() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.sender.othersender:1234", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_10() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.server.otherreceiver", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_11() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.server.otherreceiver:5678", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_12() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.receiver.port.5679", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("sender:1234", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_13() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.default", "defaultName");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("defaultName", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ClientName_14() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("clientName.default", "defaultName");
        properties.put("clientName.receiver.server.receiver:5678", "replacedSender");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedSender", fixture.getClientName());
    }

    /**
     * 
     */
    @Test
    public void test_ServiceName_01() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, null);
        
        assertEquals("/service/path", fixture.getServiceName());
    }

    /**
     * 
     */
    @Test
    public void test_ServiceName_02() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("/service/path", fixture.getServiceName());
    }

    /**
     * 
     */
    @Test
    public void test_ServiceName_03() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("serviceName.path./service/path", "replacedService");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("replacedService", fixture.getServiceName());
    }

    /**
     * 
     */
    @Test
    public void test_ServiceName_04() throws Exception {
        HttpExchange exchange = createDummyExchange();
        SOAPMessage dummyMessage = createDummySOAPMessage();
        Properties properties = new Properties();
        properties.put("serviceName.path./service/otherpath", "replacedService");
        
        SOAPEventType fixture = new SOAPEventType(exchange, dummyMessage, null, properties);
        
        assertEquals("/service/path", fixture.getServiceName());
    }

    /**
     *
     */
    private HttpExchange createDummyExchange() {
        HttpExchange result = new HttpExchange();
        result.setSender(createDummyAddress("sender", "1.2.3.4", 1234));
        result.setReceiver(createDummyAddress("receiver", "5.6.7.8", 5678));
        result.setRequest(createDummyRequest());
        return result;
    }

    /**
     * 
     */
    private Address createDummyAddress(String host, String ip, int port) {
        Address address = new Address();
        address.setHost(host);
        address.setIp(ip);
        address.setPort(BigInteger.valueOf(port));
        return address;
    }

    /**
     *
     */
    private HttpRequest createDummyRequest() {
        HttpRequest request = new HttpRequest();
        request.setUrl("http://receiver:5678/service/path");
        return request;
    }

    /**
     *
     */
    private SOAPMessage createDummySOAPMessage() throws Exception {
        MessageFactory factory = MessageFactory.newInstance();
        SOAPMessage message = factory.createMessage();
        return message;
    }


}
