//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usability;

import java.util.Collection;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.tasktrees.manager.TaskTreeManager;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * This command generates a task tree based on the provided sequences. It uses the
 * {@link TaskTreeManager} for this purpose. Please consult the documentation of the task tree
 * manager for more details.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDgenerateTaskTree implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "generateTaskTree <sequences> {<tasktree>} {-considerEventEquality} " +
            "{-minimumSequenceCoverage=<some integer>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String sequencesName = null;
        String tasktreeName = null;
        boolean useEventEqualityForTaskComparison = false;
        int minimumSequenceCoverage = 0;
        
        try {
            for (int i = 0; i < parameters.size(); i++) {
                String parameter = (String) parameters.get(i);
                if (!parameter.startsWith("-")) {
                    if (sequencesName == null) {
                        sequencesName = parameter;
                    }
                    else if (tasktreeName == null) {
                        tasktreeName = parameter;
                    }
                    else {
                        throw new IllegalArgumentException("unrecognized parameter: " + parameter);
                    }
                }
                else {
                    if ("-considerEventEquality".equals(parameter)) {
                        useEventEqualityForTaskComparison = true;
                    }
                    else if (parameter.startsWith("-minimumSequenceCoverage=")) {
                        try {
                            minimumSequenceCoverage = Integer.parseInt
                                (parameter.substring("-minimumSequenceCoverage=".length()));
                        }
                        catch (Exception e) {
                            throw new IllegalArgumentException
                                ("invalid value for parameter minimumSequenceCoverage: " +
                                 parameter.substring("-minimumSequenceCoverage=".length()));
                        }
                    }
                    else {
                        throw new IllegalArgumentException("unrecognized parameter: " + parameter);
                    }
                }
            }
        }
        catch (IllegalArgumentException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IllegalArgumentException("could not process parameters", e);
        }
        
        if (sequencesName == null) {
            throw new IllegalArgumentException("must provide a sequences name");
        }
        
        if (tasktreeName == null) {
            tasktreeName = "tasktree";
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;
        
        ITaskModel taskModel = new TaskTreeManager().createTaskModel
            (sequences, useEventEqualityForTaskComparison, minimumSequenceCoverage);
        
        if (GlobalDataContainer.getInstance().addData(tasktreeName, taskModel)) {
            CommandHelpers.dataOverwritten(tasktreeName);
        }
        
    }

}
