//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.eventcore;

import java.io.Serializable;

/**
 * <p>
 * Common interface for event targets. An event target can, e.g., be an element of a GUI or Web
 * server or a VR. A concrete event-driven software platform can define its event targets through
 * the implementation of this interface. This type of event target differ from those of the parent
 * type in that they can build tree structures such as a {@link IHierarchicalEventTargetModel}
 * </p>
 * 
 * @version 1.0
 * @author Steffen Herbold
 */
public interface IHierarchicalEventTarget extends IEventTarget, Serializable {
    
    /**
     * <p>
     * Returns the parent of the event target.
     * </p>
     * 
     * @return the parent
     */
    public IHierarchicalEventTarget getParent();

    /**
     * <p>
     * Returns the specification of the event target.
     * </p>
     * 
     * @return the specification
     */
    public IEventTargetSpec getSpecification();

    /**
     * <p>
     * Updates the specification of a event target with another specification, e.g., to add further
     * known names of the event target.
     * </p>
     * 
     * @param furtherSpec
     *            additional specification
     */
    public void updateSpecification(IEventTargetSpec furtherSpec);

    /**
     * <p>
     * returns the event target model to which this event target belongs (if any).
     * </p>
     *
     * @return as described
     */
    public IHierarchicalEventTargetModel<? extends IHierarchicalEventTarget> getEventTargetModel();

    /**
     * <p>
     * The {@link IHierarchicalEventTarget} that is passed by this function is equal to the current
     * event target and will hereafter be treated as such.
     * </p>
     * 
     * @param equalTarget
     *            event target that is equal
     */
    public void addEqualEventTarget(IHierarchicalEventTarget equalTarget);
}
