//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.genericevents;

import java.util.Map;

import org.xml.sax.SAXException;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.EventTargetModelException;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.plugin.genericevents.eventCore.GenericEventTarget;
import de.ugoe.cs.autoquest.plugin.genericevents.eventCore.GenericEventTargetSpec;

/**
 * <p>
 * This class provides the functionality to parse XML log files generated by the generic event
 * monitor of AutoQUEST. The result of parsing a file is a collection of event sequences and a
 * target model.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 * 
 */
public class GenericEventLogParser extends AbstractDefaultLogParser {
    
    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.plugin.html.AbstractDefaultLogParser#handleGUIElement(String, Map)
     */
    @Override
    protected boolean handleTarget(String id, Map<String, String> parameters)
        throws SAXException
    {
        String parentId = parameters.get("parent");
        
        GenericEventTargetSpec specification = new GenericEventTargetSpec(id, parameters);
        
        try {
            super.getTargetTree().add(id, parentId, specification);
        }
        catch (EventTargetModelException e) {
            throw new SAXException("could not handle generic event target with id " +
                                   id + ": " + e.getMessage(), e);
        }
        
        return true;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.plugin.html.AbstractDefaultLogParser#handleEvent(String, Map)
     */
    @Override
    protected boolean handleEvent(String type, Map<String, String> parameters) throws SAXException {
        String targetId = parameters.get("targetId");
        
        if (targetId == null) {
            throw new SAXException("event does not have a target id");
        }
        
        GenericEventTarget target = super.getTargetTree().find(targetId);

        IEventType eventType = new StringEventType(type);
        
        if (eventType != null) {
            Event event = new Event(eventType, target);

            String timestampStr = parameters.get("timestamp");
        
            if (timestampStr != null) {
                event.setTimestamp(Long.parseLong(timestampStr));
            }
        
            super.addToSequence(event);
        }
        // else ignore unknown event type

        return true;
    }

}
