//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.


package de.ugoe.cs.autoquest.ui.swt;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.List;

import de.ugoe.cs.autoquest.eventcore.IHierarchicalEventTarget;
import de.ugoe.cs.autoquest.eventcore.IHierarchicalEventTargetModel;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * listing GUI models contained in the data store
 * </p>
 * 
 * @author 2012, Steffen Herbold, Patrick Harms
 */
public class GuiModelTabComposite extends Composite {

    List guiModelList;

    /**
     * Create the composite.
     */
    public GuiModelTabComposite(Composite parent, int style) {
        super(parent, style);
        createContents();
    }

    /**
     * <p>
     * fills the list of available GUI models and provides buttons for their detailed visualization
     * </p>
     */
    private <T extends IHierarchicalEventTarget> void createContents() {
        setLayout(new GridLayout(5, false));

        guiModelList = new List(this, SWT.BORDER | SWT.V_SCROLL);
        guiModelList.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 5, 1));

        Button btnShow = new Button(this, SWT.NONE);
        btnShow.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                String[] selectedStrings = guiModelList.getSelection();
                if (selectedStrings.length == 0) {
                    SWTHelpers.noSelectionError(getShell());
                    return;
                }
                String modelName = selectedStrings[0];
                
                @SuppressWarnings("unchecked")
                IHierarchicalEventTargetModel<T> model =
                    (IHierarchicalEventTargetModel<T>) GlobalDataContainer.getInstance().getData(modelName);

                ShowGuiModelDialog<T> showGuiModelDialog =
                    new ShowGuiModelDialog<T>(getShell(), SWT.NONE, model, modelName);
                
                showGuiModelDialog.open();
            }
        });
        btnShow.setText("Show");

        Button btnDelete_1 = new Button(this, SWT.NONE);
        btnDelete_1.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (SWTHelpers.deleteSelectedFromStorage(guiModelList)) {
                    updateModelList();
                }
                else {
                    SWTHelpers.noSelectionError(getShell());
                }
            }
        });
        btnDelete_1.setText("Delete");
    }

    /**
     * Disable the check that prevents subclassing of SWT components
     */
    @Override
    protected void checkSubclass() {
        // Disable the check that prevents subclassing of SWT components
    }

    /**
     * <p>
     * convenience method for updating the list of available GUI models
     * </p>
     */
    public void updateModelList() {
        guiModelList.removeAll();
        for(String key : GlobalDataContainer.getInstance().getAllKeys()) {
            if( GlobalDataContainer.getInstance().getData(key) instanceof IHierarchicalEventTargetModel ) {
                guiModelList.add(key);
            }
        }
    }

}
