//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.genericeventmonitor;

import java.util.Map;

/**
 * <p>
 * represents an event caused by a user on a client. An event contains the infos
 * about the client ({@link ClientInfos}, when and where the event took place, the type of
 * event and some additional infos such as the event coordinates or the number of the pressed
 * key. 
 * </p>
 * 
 * @author Patrick Harms
 */
class GenericEvent {

    /**
     * infos about the client that caused the event
     */
    private ClientInfos clientInfos;

    /**
     * the time stamp of the event
     */
    private Long time;

    /**
     * the target on which the event was executed
     */
    private GenericEventTarget target;

    /**
     * the type of the event, e.g. onclick
     */
    private String eventType;

    /**
     * additional parameters of the event
     */
    private Map<String, String> parameters;

    /**
     * <p>
     * initializes the event with all relevant infos
     * </p>
     *
     * @param clientInfos    infos about the client that caused the event
     * @param time           the time stamp of the event
     * @param target         the HTML element on which the event was executed
     * @param eventType      the type of the event
     * @param parameters     the parameters of the event
     */
    GenericEvent(ClientInfos         clientInfos,
                 Long                time,
                 GenericEventTarget  target,
                 String              eventType,
                 Map<String, String> parameters)
    {
        this.clientInfos = clientInfos;
        this.time = time;
        this.target = target;
        this.eventType = eventType;
        this.parameters = parameters;
    }

    /**
     * @return the clientInfos
     */
    ClientInfos getClientInfos() {
        return clientInfos;
    }

    /**
     * @return the time
     */
    Long getTime() {
        return time;
    }

    /**
     * @return the target
     */
    GenericEventTarget getTarget() {
        return target;
    }

    /**
     * @return the eventType
     */
    String getEventType() {
        return eventType;
    }

    /**
     * @return the parameters
     */
    Map<String, String> getParameters() {
        return parameters;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return eventType + " on " + target;
    }
}
