//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;

import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskInstanceTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class RequiredInefficientActionsRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        Map<ISequence, double[]> smellingTasks =
            getInefficientActionStatistics(taskModel.getTasks());
        
        analyzeTasksWithInefficientActions(smellingTasks, results, taskModel);

        return results;
    }

    /**
     *
     */
    private void analyzeTasksWithInefficientActions(Map<ISequence, double[]>  smellingTasks,
                                                    UsabilityEvaluationResult results,
                                                    ITaskModel                taskModel)
    {

        for (Map.Entry<ISequence, double[]> entry : smellingTasks.entrySet()) {
            DescriptiveStatistics stats = new DescriptiveStatistics(entry.getValue());
            
            int ratio = (int) (1000 * stats.getMean());

            UsabilitySmellIntensity severity =
                UsabilitySmellIntensity.getIntensity(ratio, entry.getKey(), taskModel);

            if (severity != null) {
                Map<String, Object> parameters = new HashMap<String, Object>();
                parameters.put("task", entry.getKey());
                parameters.put("ratio", (ratio / 10));

                results.addSmell(entry.getKey(), severity,
                                 UsabilitySmellDescription.INEFFICIENT_ACTIONS, parameters);
            }
        }
    }

    /**
     * 
     */
    private Map<ISequence, double[]> getInefficientActionStatistics(Collection<ITask> tasks) {
        Map<ISequence, double[]> inefficientActionRatios = new HashMap<ISequence, double[]>();
        
        for (ITask task : tasks) {
            if (task instanceof ISequence)  {
                if (countEfficientActions((ISequence) task) > 1) {
                    double[] ratios = getRatiosOfInefficientActionsInInstances((ISequence) task);
                    
                    for (int i = 0; i < ratios.length; i++) {
                        if (ratios[i] > 0) {
                            // there is at least on inefficient action
                            inefficientActionRatios.put((ISequence) task, ratios);
                            break;
                        }
                    }
                }
            }
        }
        
        return inefficientActionRatios;
    }

    /**
     * 
     */
    private int countEfficientActions(ISequence task) {
        final List<IEventTask> efficientActions = new LinkedList<>();
        
        task.accept(new DefaultTaskTraversingVisitor() {
            @Override
            public void visit(IEventTask eventTask) {
                if (!ActionClassifier.isInefficient(eventTask)) {
                    efficientActions.add(eventTask);
                }
            }
        });
        
        return efficientActions.size();
    }

    /**
     *
     */
    private double[] getRatiosOfInefficientActionsInInstances(ISequence sequence) {
        Collection<ITaskInstance> instances = sequence.getInstances();
        
        double[] ratios = new double[instances.size()];
        int index = 0;
        for (ITaskInstance instance : instances) {
            ratios[index++] = getRatioOfInefficientActionsInInstance(instance);
        }
        
        return ratios;
    }

    /**
     *
     */
    private double getRatioOfInefficientActionsInInstance(ITaskInstance instance) {
        final int[] count = new int[2];
        count[0] = 0;
        count[1] = 0;
        
        instance.accept(new DefaultTaskInstanceTraversingVisitor() {
            @Override
            public void visit(IEventTaskInstance eventTaskInstance) {
                if (ActionClassifier.isInefficient(eventTaskInstance.getEvent())) {
                    count[0]++;
                }
                
                count[1]++;
            }
            
        });
        
        return (double) count[0] / count[1];
    }

}
