//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import static de.ugoe.cs.autoquest.usability.UsabilitySmellDescription.TEXT_FIELD_INPUT_RATIO;
import static de.ugoe.cs.autoquest.usability.UsabilitySmellDescription.TEXT_FIELD_INPUT_REPETITIONS;
import static de.ugoe.cs.autoquest.usability.UsabilitySmellDescription.TEXT_FIELD_NO_LETTER_OR_DIGIT_RATIO;

import org.junit.Test;

import de.ugoe.cs.autoquest.usability.UsabilitySmell;

/**
 *
 */
public class TextInputStatisticsRuleTest extends AbstractUsabilityEvaluationTC {

    /**
     *
     */
    @Test
    public void testWithDifferentTextInputInteractionsOnly_01() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  TextInput elem1 (bla) {}" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithDifferentTextInputInteractionsOnly_02() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  TextInput elem1 (a) {}" +
            "  TextInput elem1 (b) {}" +
            "  TextInput elem1 (c) {}" +
            "  TextInput elem1 (d) {}" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testWithDifferentTextInputInteractionsOnly_03() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  TextInput elem1 (a) {}" +
            "  TextInput elem1 (b) {}" +
            "  TextInput elem1 (c) {}" +
            "  TextInput elem1 (d) {}" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testWithDifferentTextInputInteractionsOnly_04() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Iteration {" +
            "    TextInput elem1 (bla) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testWithDifferentTextInputInteractionsOnly_05() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    TextInput elem1 (a) {}" +
            "    Sequence seq1 {" +
            "      TextInput elem1 (b) {}" +
            "      TextInput elem1 (c) {}" +
            "      TextInput elem1 (d) {}" +
            "      TextInput elem1 (e) {}" +
            "    }" +
            "    Iteration it1 {" +
            "      TextInput elem1 (f) {}" +
            "     TextInput elem1 (g) {}" +
            "      TextInput elem1 (h) {}" +
            "      TextInput elem1 (i) {}" +
            "    }" +
            "    TextInput elem1 (j) {}" +
            "    Selection sel1 {" +
            "      TextInput elem1 (k) {}" +
            "    }" +
            "    Sequence seq2 {" +
            "      TextInput elem1 (l) {}" +
            "      Sequence seq1 {" +
            "        TextInput elem1 (m) {}" +
            "        TextInput elem1 (n) {}" +
            "        TextInput elem1 (o) {}" +
            "        TextInput elem1 (p) {}" +
            "      }" +
            "      Iteration it1 {" +
            "        TextInput elem1 (q) {}" +
            "        TextInput elem1 (r) {}" +
            "        TextInput elem1 (s) {}" +
            "        TextInput elem1 (t) {}" +
            "      }" +
            "      TextInput elem1 (u) {}" +
            "      Selection sel1 {" +
            "        TextInput elem1 (v) {}" +
            "      }" +
            "    }" +
            "    Selection sel2 {" +
            "      TextInput elem1 (w) {}" +
            "    }" +
            "    TextInput elem1 (x) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_01() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem1 (b) {}" +
            "    Interaction elem1 {}" +
            "    TextInput elem1 (c) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_02() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    TextInput elem1 (a) {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    TextInput elem1 (c) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_03() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    TextInput elem1 (a) {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[0];

        //assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_04() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Selection {" +
            "    TextInput elem1 (a) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_05() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    Sequence seq2 {" +
            "      Interaction elem1 {}" +
            "      TextInput elem1 (b) {}" +
            "      TextInput elem1 (c) {}" +
            "      Interaction elem1 {}" +
            "      TextInput elem1 (d) {}" +
            "    }" +
            "    Iteration it1 {" +
            "      TextInput elem1 (e) {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "    Selection sel1 {" +
            "      TextInput elem1 (f) {}" +
            "    }" +
            "    Sequence seq3 {" +
            "      TextInput elem1 (j) {}" +
            "      Sequence seq4 {" +
            "        TextInput elem1 (k) {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (l) {}" +
            "        TextInput elem1 (m) {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (n) {}" +
            "        TextInput elem1 (o) {}" +
            "      }" +
            "      Iteration it2{" +
            "        Interaction elem1 {}" +
            "      }" +
            "      Interaction elem1 {}" +
            "      Selection sel1 {" +
            "        TextInput elem1 (p) {}" +
            "      }" +
            "    }" +
            "    Selection sel1 {" +
            "      Sequence seq 5{" +
            "        TextInput elem1 (w) {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (x) {}" +
            "        TextInput elem1 (y) {}" +
            "        Interaction elem1 {}" +
            "      }" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }


    /**
     *
     */
    @Test
    public void testCombinationsOfTextInputInteractionsAndOtherInteractions_06() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    Sequence seq2 {" +
            "      Interaction elem1 {}" +
            "      TextInput elem1 (b) {}" +
            "      Interaction elem1 {}" +
            "      Interaction elem1 {}" +
            "      TextInput elem1 (c) {}" +
            "    }" +
            "    Iteration it1 {" +
            "      TextInput elem1 (d) {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "    Selection sel1 {" +
            "      TextInput elem1 (e) {}" +
            "    }" +
            "    Sequence seq3 {" +
            "      TextInput elem1 (i) {}" +
            "      Sequence seq4 {" +
            "        TextInput elem1 (j) {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (k) {}" +
            "        Interaction elem1 {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (m) {}" +
            "        Interaction elem1 {}" +
            "      }" +
            "      Iteration it2 {" +
            "        Interaction elem1 {}" +
            "      }" +
            "      Interaction elem1 {}" +
            "      Selection sel1 {" +
            "        Interaction elem1 {}" +
            "      }" +
            "    }" +
            "    Selection sel1 {" +
            "      Sequence seq5 {" +
            "        TextInput elem1 (v) {}" +
            "        Interaction elem1 {}" +
            "        Interaction elem1 {}" +
            "        TextInput elem1 (x) {}" +
            "        Interaction elem1 {}" +
            "      }" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_01() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem2 (a) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_02() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    Sequence seq2 {" +
            "      TextInput elem2 (a) {}" +
            "    }" +
            "    Iteration it1 {" +
            "      TextInput elem3 (a) {}" +
            "    }" +
            "    Selection sel1 {" +
            "      TextInput elem4 (a) {}" +
            "    }" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_03() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem2 (z) {}" +
            "    TextInput elem1 (b) {}" +
            "    TextInput elem2 (b) {}" +
            "    TextInput elem1 (c) {}" +
            "    TextInput elem2 (c) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_04() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem2 (z) {}" +
            "    TextInput elem1 (b) {}" +
            "    TextInput elem2 (y) {}" +
            "    TextInput elem1 (c) {}" +
            "    TextInput elem2 (c) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_05() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem2 (z) {}" +
            "    TextInput elem1 (b) {}" +
            "    TextInput elem2 (y) {}" +
            "    TextInput elem1 (c) {}" +
            "    TextInput elem2 (x) {}" +
            "    TextInput elem1 (d) {}" +
            "    TextInput elem2 (w) {}" +
            "    TextInput elem1 (e) {}" +
            "    TextInput elem2 (v) {}" +
            "    TextInput elem1 (f) {}" +
            "    TextInput elem2 (f) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_06() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    TextInput elem1 (b) {}" +
            "    Sequence seq2 {" +
            "      TextInput elem2 (a) {}" +
            "      TextInput elem2 (b) {}" +
            "      TextInput elem2 (c) {}" +
            "    }" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testTextEntryRepetitions_07() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    TextInput elem1 (a) {}" +
            "    Sequence seq2 {" +
            "      TextInput elem2 (a) {}" +
            "      TextInput elem2 (b) {}" +
            "      TextInput elem2 (c) {}" +
            "    }" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS),
              new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_REPETITIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testNoLetterOrDigitInput_01() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    TextInput elem1 (_a_b_c_) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_NO_LETTER_OR_DIGIT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testNoLetterOrDigitInput_02() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    TextInput elem1 (12345_6789012345) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_NO_LETTER_OR_DIGIT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testNoLetterOrDigitInput_03() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    TextInput elem1 (123456789012345678901234567890_123456789012345) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_NO_LETTER_OR_DIGIT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testNoLetterOrDigitInput_04() {
        TextInputStatisticsRule rule = new TextInputStatisticsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    TextInput elem1 (1234567890123456789012345678901234567890123456789_01234567890" +
            "12345678901234567890123456789012345) {}" +
            "  }" +
            "}";

        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_INPUT_RATIO),
              new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  TEXT_FIELD_NO_LETTER_OR_DIGIT_RATIO) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }
}
