//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.uml;

import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.Package;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Stereotype;

/**
 * <p>
 * Helper functions for working with the UML Testing Profile (UTP).
 * </p>
 * 
 * @author Steffen Herbold
 */
public class UTPUtils {

    /**
     * <p>
     * Retrieves the UTP profile from a UML {@link Package}, if it is applied.
     * </p>
     * 
     * @param umlModel
     *            model where the profile is retrieved from or added to
     * @return UTP profile
     */
    public static Profile getOrApplyUtpProfile(Model umlModel) {
        // TODO apply UTP if it is missing
        return umlModel.getAppliedProfile("utp");
    }

    /**
     * <p>
     * Retrieves the UTP TestCase stereotype. In case UTP is not yet applied to the model, it is
     * applied automatically.
     * </p>
     * 
     * @param umlModel
     *            model for the which the UTP TestCase stereotype is retrieved
     * @return UTP TestCase stereotype
     */
    public static Stereotype getTestCaseStereotype(Model umlModel) {
        return (Stereotype) getOrApplyUtpProfile(umlModel).getOwnedMember("TestCase");
    }

    /**
     * <p>
     * Retrieves the UTP TestComponent stereotype. In case UTP is not yet applied to the model, it
     * is applied automatically.
     * </p>
     * 
     * @param umlModel
     *            model for the which the UTP TestComponent stereotype is retrieved
     * @return UTP TestComponent stereotype
     */
    public static Stereotype getTestComponentStereotype(Model umlModel) {
        return (Stereotype) getOrApplyUtpProfile(umlModel).getOwnedMember("TestComponent");
    }

    /**
     * <p>
     * Retrieves the UTP TestContext stereotype. In case UTP is not yet applied to the model, it is
     * applied automatically.
     * </p>
     * 
     * @param umlModel
     *            model for the which the UTP TestContext stereotype is retrieved
     * @return UTP TestContext stereotype
     */
    public static Stereotype getTestContextStereotype(Model umlModel) {
        return (Stereotype) getOrApplyUtpProfile(umlModel).getOwnedMember("TestContext");
    }

    /**
     * <p>
     * Retrieves the UTP SUT stereotype. In case UTP is not yet applied to the model, it is applied
     * automatically.
     * </p>
     * 
     * @param umlModel
     *            model for the which the UTP SUT stereotype is retrieved
     * @return UTP SUT stereotype
     */
    public static Stereotype getSUTStereotype(Model umlModel) {
        return (Stereotype) getOrApplyUtpProfile(umlModel).getOwnedMember("SUT");
    }

    /**
     * <p>
     * Retrieves the UTP LiteralAny stereotype. In case UTP is not yet applied to the model, it is
     * applied automatically.
     * </p>
     * 
     * @param umlModel
     *            model for the which the UTP LiteralAny stereotype is retrieved
     * @return UTP LiteralAny stereotype
     */
    public static Stereotype getLiteralAnyStereotype(Model umlModel) {
        return (Stereotype) getOrApplyUtpProfile(umlModel).getOwnedMember("LiteralAny");
    }

}
