//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.sequences;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.SequenceInstanceOf;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to merge all sequences of a collection of sequences into one long sequence depending on
 * the timestamps of the events
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDmergeSequences implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String sequencesName;
        String newSequencesName;
        try {
            sequencesName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                newSequencesName = (String) parameters.get(1);
            }
            else {
                newSequencesName = sequencesName;
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("must provide a sequences name");
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;
        
        List<Event> resultingSequence = new LinkedList<Event>();
        long first = 0;
        long last = 0;
        
        Console.println("merging " + sequences.size() + " sequences");
        for (List<Event> sequence : sequences) {
            for (Event event : sequence) {
                if (event.getTimestamp() < 0) {
                    Console.printerrln("at least one of the events does not have a correct " +
                                       "timestamp required for the merging of sequences and " +
                                       "sorting of events");
                }
                
                if (resultingSequence.isEmpty()) {
                    resultingSequence.add(event);
                    first = event.getTimestamp();
                    last = event.getTimestamp();
                }
                else if (event.getTimestamp() < first) {
                    resultingSequence.add(0, event);
                    first = event.getTimestamp();
                }
                else if (event.getTimestamp() > last) {
                    resultingSequence.add(event);
                    last = event.getTimestamp();
                }
                else {
                    for (int i = 0; i < resultingSequence.size(); i++) {
                        if (event.getTimestamp() <= resultingSequence.get(i).getTimestamp()) {
                            resultingSequence.add(i, event);
                            break;
                        }
                    }
                }
            }
        }
        
        Collection<List<Event>> newSequences = new LinkedList<List<Event>>();
        newSequences.add(resultingSequence);
        
        if (GlobalDataContainer.getInstance().addData(newSequencesName, newSequences)) {
            CommandHelpers.dataOverwritten(newSequencesName);
        }
        
        Console.println("sequences merged");
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "mergeSequences <sequencesName> {<new sequences>}";
    }

}
