//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.genericevents;

import java.time.Duration;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.Set;

import org.xml.sax.SAXException;

import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.EventTargetModelException;
import de.ugoe.cs.autoquest.eventcore.IEventType;
import de.ugoe.cs.autoquest.eventcore.StringEventType;
import de.ugoe.cs.autoquest.plugin.genericevents.eventCore.GenericEventTarget;
import de.ugoe.cs.autoquest.plugin.genericevents.eventCore.GenericEventTargetSpec;

/**
 * <p>
 * This class provides the functionality to parse XML log files generated by the generic event
 * monitor of AutoQUEST. The result of parsing a file is a collection of event sequences and a
 * target model.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 * 
 */
public class GenericEventLogParser extends AbstractDefaultLogParser {
    
    /** types of events to be ignored */
    private Set<String> ignoredEvents;
    
    
    /**
     * <p>
     * used to provide the parser with a set of ignored event types.
     * </p>
     *
     * @param ignoredEvents
     */
    public GenericEventLogParser(Set<String> ignoredEvents) {
        super();
        this.ignoredEvents = ignoredEvents;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.plugin.html.AbstractDefaultLogParser#handleGUIElement(String, Map)
     */
    @Override
    protected boolean handleTarget(String id, Map<String, String> parameters)
        throws SAXException
    {
        String parentId = parameters.get("parent");
        
        if (parentId != null) {
            // check, if the parent is already in the tree
            if (super.getTargetTree().find(parentId) == null) {
                // element cannot yet be processed as parent is not existing yet
                return false;
            }
        }
        
        GenericEventTargetSpec specification = new GenericEventTargetSpec(id, parameters);
        
        try {
            super.getTargetTree().add(id, parentId, specification);
        }
        catch (EventTargetModelException e) {
            throw new SAXException("could not handle generic event target with id " +
                                   id + ": " + e.getMessage(), e);
        }
        
        return true;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.plugin.html.AbstractDefaultLogParser#handleEvent(String, Map)
     */
    @Override
    protected boolean handleEvent(String type, Map<String, String> parameters) throws SAXException {
        // ignore the event based on its type if requested
        if (ignoredEvents.contains(type)) {
            return true;
        }
        
        String targetId = parameters.get("targetId");
        
        // ignore the event based on type and target id if requested
        if (ignoredEvents.contains(type + "." + targetId)) {
            return true;
        }
        
        if (targetId == null) {
            throw new SAXException("event does not have a target id");
        }
        
        GenericEventTarget target = super.getTargetTree().find(targetId);

        IEventType eventType = new StringEventType(type);
        
        if (eventType != null) {
            Event event = new Event(eventType, target);

            String timestampStr = parameters.get("timestamp");
        
            if (timestampStr != null) {
                event.setTimestamp(determineTimestamp(timestampStr));
            }
            
            for (Map.Entry<String, String> parameter : parameters.entrySet()) {
                if (!"targetId".equals(parameter.getKey()) &&
                    !"timestamp".equals(parameter.getKey()))
                {
                    event.setParameter(parameter.getKey(), parameter.getValue());
                }
            }
        
            super.addToSequence(event);
        }
        // else ignore unknown event type

        return true;
    }

    /**
     * convenience method to align different timestamp formats
     */
    private long determineTimestamp(String timestampStr) {
        long val = Long.parseLong(timestampStr);
        
        // We expect any recording to have taken place in years with at most 4 digits. Hence,
        // divide the val until we reach such a year
        long max = new GregorianCalendar(10000, 1, 1).getTimeInMillis();
        
        while (max < val) {
            val /= 10; 
        }
        
        // now, it can still be the case, that the base line is wrong. I.e., the remaining value
        // may be milliseconds, but not starting from 1970, as we expect it, but starting from
        // year 1 of the gregorian calendar. If this is the case, the date is still in the future.
        // Hence, we substract the milliseconds between year 1 of the gregorian calendar and 1970.
        // (We took these magic number, as they provide correct results. It's not fully clear,
        //  why the January 1st 1970 does not work correctly but provides an offset of two days)
        if (val > System.currentTimeMillis()) {
            Duration duration = Duration.between(new GregorianCalendar(1, 1, 1).toInstant(),
                                                 new GregorianCalendar(1969, 12, 30).toInstant());
            
            val -= duration.toMillis();
        }
        
        return val;
    }

}
