//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.guimodel;

import de.ugoe.cs.autoquest.eventcore.IEventTargetSpec;
import de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec;

/**
 * <p>
 * a GUI element specification of an HTML server. This is the root element for a GUI model of
 * an HTML web site. It is identified through a host and a port. Its children are documents.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HTMLServerSpec extends HTMLGUIElementSpec implements IGUIElementSpec {

    /**
     * <p>
     * default serial version UID
     * </p>
     */
    private static final long serialVersionUID = 1L;
    
    /**
     * <p>
     * the host of the server
     * </p>
     */
    private String host;
    
    /**
     * <p>
     * the port of the server
     * </p>
     */
    private int port;

    /**
     * <p>
     * initialize the server specification with a host and a port.
     * </p>
     *
     * @param host the host name of the server
     * @param port the port number of the server
     * 
     * @throws IllegalArgumentException if one of the parameters is invalid (e.g. host = null)
     */
    public HTMLServerSpec(String host, int port) {
        super("server", (host != null ? host.hashCode() : 0) + port);
        
        if (host == null) {
            throw new IllegalArgumentException("host must not be null");
        }
        
        if ((port < -1) || (port > 65536)) {
            throw new IllegalArgumentException("port " + port + " is not a valid port");
        }

        this.host = host;
        this.port = port;
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.eventcore.guimodel.IGUIElementSpec#getSimilarity(IGUIElementSpec)
     */
    @Override
    public boolean getSimilarity(IEventTargetSpec other) {
        if (other instanceof HTMLServerSpec) {
            HTMLServerSpec otherSpec = (HTMLServerSpec) other;
            
            if (!super.getSimilarity(otherSpec)) {
                return false;
            }
            else if (!host.equals(otherSpec.host)) {
                return false;
            }
            else {
                return (port == otherSpec.port);
            }
        }
        
        return false;
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        String str = "Server(" + getHost();

        if (getPort() > -1 ) {
            str += ":" + getPort();
        }
        
        str += ")";
        return str;
    }

    /**
     * <p>
     * returns the host of the represented server
     * </p>
     *
     * @return the host
     */
    String getHost() {
        return this.host;
    }

    /**
     * <p>
     * returns the port of the represented server
     * </p>
     *
     * @return the port
     */
    int getPort() {
        return this.port;
    }

}
