//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.eventcore;

import de.ugoe.cs.autoquest.keyboardmaps.VirtualKey;

/**
 * <p>
 * Map of virtual keys for HTML.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public enum HTMLVirtualKey {
    VK_BACKSPACE(0x08, VirtualKey.BACK_SPACE),
    VK_TAB(0x09, VirtualKey.TAB),
    VK_RETURN(0x0D, VirtualKey.ENTER);
    
    /**
     * <p>
     * Numerical representation of the virtual key.
     * </p>
     */
    private int mNumber;

    /**
     * <p>
     * {@link VirtualKey} represented by this HTMLVirtualKey
     * </p>
     */
    private VirtualKey mRepresentedKey;

    /**
     * <p>
     * Constructor. Creates a new HTMLVirtualKey.
     * </p>
     * 
     * @param number
     *            numerical representation of the virtual key
     * @param representedKey
     *            virtual key that is represented
     */
    HTMLVirtualKey(int number, VirtualKey representedKey) {
        mNumber = number;
        mRepresentedKey = representedKey;
    }

    /**
     * <p>
     * Returns the numerical representation of the virtual key.
     * </p>
     * 
     * @return the numerical representation
     */
    int getNumber() {
        return mNumber;
    }

    /**
     * <p>
     * Parses an {@link String} and returns the respective HTMLVirtualKey if possible.
     * </p>
     * 
     * @param string
     *            String representation of the event type
     * @return created HTMLVirtualKey
     * @throws IllegalArgumentException
     *             thrown if there is no HTMLVirtualKey that correlates to string
     */
    public static HTMLVirtualKey parseVirtualKey(String string) throws IllegalArgumentException {
        for (HTMLVirtualKey virtualKey : HTMLVirtualKey.values()) {
            if (virtualKey.mNumber == Integer.parseInt(string)) {
                return virtualKey;
            }
        }

        throw new IllegalArgumentException("there is no virtual key with id " + string);
    }

    /**
     * <p>
     * Returns the HTMLVirtualKey associated with an integer.
     * </p>
     * 
     * @param number
     *            integer to which the according HTMLVirtualKey is returned
     * @return the HTMLVirtualKey
     * @throws IllegalArgumentException
     *             thrown if there is no HTMLVirtualKey that correlates to number
     */
    public static HTMLVirtualKey valueOf(int number) throws IllegalArgumentException {
        for (HTMLVirtualKey virtualKey : HTMLVirtualKey.values()) {
            if (virtualKey.mNumber == number) {
                return virtualKey;
            }
        }

        throw new IllegalArgumentException("there is no virtual key with number " + number);
    }

    /**
     * <p>
     * Returns the {@link VirtualKey} associated with this HTMLVirtualKey.
     * </p>
     * 
     * @return the virtual key
     */
    public VirtualKey getKey() {
        return mRepresentedKey;
    }

}
