//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import de.ugoe.cs.autoquest.tasktrees.treeifc.DefaultTaskTraversingVisitor;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIteration;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IIterationInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ISequence;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskModel;

/**
 * TODO comment
 * 
 * @version $Revision: $ $Date: 16.07.2012$
 * @author 2012, last modified by $Author: pharms$
 */
public class TaskRetryRule implements UsabilityEvaluationRule {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.usability.UsabilityEvaluationRule#evaluate(TaskTree)
     */
    @Override
    public UsabilityEvaluationResult evaluate(ITaskModel taskModel) {
        UsabilityEvaluationResult results = new UsabilityEvaluationResult(taskModel);

        Map<IIteration, Integer> smellingTasks = getTasksWithRetries(taskModel.getTasks());
        analyzeTasksWithRetries(smellingTasks, results, taskModel);

        return results;
    }

    /**
     *
     */
    private void analyzeTasksWithRetries(Map<IIteration, Integer>  smellingTasks,
                                         UsabilityEvaluationResult results,
                                         ITaskModel                taskModel)
    {

        for (Map.Entry<IIteration, Integer> entry : smellingTasks.entrySet()) {
            UsabilitySmellIntensity intensity =
                UsabilitySmellIntensity.getIntensity(entry.getValue(), entry.getKey(), taskModel);

            if (intensity != null) {
                Map<String, Object> parameters = new HashMap<String, Object>();

                int instances = 0;
                int numberOfRepeatedInstances = 0;
                int cummulativeRepetitions = 0;
                
                for (ITaskInstance instance : entry.getKey().getInstances()) {
                    instances++;
                    
                    if (((IIterationInstance) instance).size() > 1) {
                        numberOfRepeatedInstances++;
                        cummulativeRepetitions += ((IIterationInstance) instance).size() - 1;
                    }
                }
                
                parameters.put("repeatedInstanceRatio",
                               (((float) instances) / numberOfRepeatedInstances));
                parameters.put("averageRepetitionRatio",
                               (((float) cummulativeRepetitions) / numberOfRepeatedInstances));
                parameters.put("task", entry.getKey().getMarkedTask());
                
                results.addSmell(entry.getKey().getMarkedTask(), intensity,
                                 UsabilitySmellDescription.TASK_RETRIED, parameters);
            }
        }
    }

    /**
     * 
     */
    private Map<IIteration, Integer> getTasksWithRetries(Collection<ITask> tasks) {
        Map<IIteration, Integer> retryRatios = new HashMap<IIteration, Integer>();
        
        for (ITask task : tasks) {
            if (isTaskOfConsideration(task))  {
                int rate = getRetryRate((IIteration) task);
                
                if (rate > 0) {
                    retryRatios.put((IIteration) task, rate);
                }
            }
        }
        
        return retryRatios;
    }

    /**
     * check if a task must be considered. A task is relevant, if it is an iteration of a sequence
     * which has at least two non-inefficient actions as leaf nodes.
     */
    private boolean isTaskOfConsideration(ITask task) {
        if (!(task instanceof IIteration)) {
            return false;
        }
        
        if (!(((IIteration) task).getMarkedTask() instanceof ISequence)) {
            return false;
        }
        
        ISequence childTask = (ISequence) ((IIteration) task).getMarkedTask();
        
        if ((childTask.getInstances() != null) && (childTask.getInstances().size() > 0)) {
            return getSemanticActions(childTask).size() > 1;
        }
        else {
            return false;
        }
    }

    /**
     * 
     */
    private List<IEventTask> getSemanticActions(ITask task) {
        final List<IEventTask> semanticEventTasks = new LinkedList<>();

        task.accept(new DefaultTaskTraversingVisitor() {
            @Override
            public void visit(IEventTask eventTask) {
                if (!ActionClassifier.isInefficient(eventTask)) {
                    semanticEventTasks.add(eventTask);
                }
            }

        });

        return semanticEventTasks;
    }

    /**
     *
     */
    private int getRetryRate(IIteration task) {
        if (task.getInstances().size() > 0) {
            int numberOfRepetitions = 0;
            for (ITaskInstance instance : task.getInstances()) {
                numberOfRepetitions += ((IIterationInstance) instance).size() - 1;
            }
            
            return numberOfRepetitions * getSemanticActions(task).size();
        }
        else {
            return 0;
        }
    }
}
