//   Copyright 2015 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.genericeventmonitor;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * a generic representation of event target. An event target has an id, optionally a parent or a
 * list of children, and parameters.
 * </p>
 * 
 * @author Patrick Harms
 */
class GenericEventTarget {

    /** the id of the target */
    private String targetId;

    /** the parameters of the target providing further important target details */
    private Map<String, String> parameters;

    /** if available the parent of the given target (used for creating target hierarchies) */
    private GenericEventTarget parent;

    /** if available the children of the given target (used for creating target hierarchies) */
    private List<GenericEventTarget> children = null;

    /**
     * <p>
     * initialized the generic event target with an id, the parameters and a parent if any. All
     * parameters may be null.
     * </p>
     *
     * @param targetId   the id of the target
     * @param parameters the parameters to provide further target details
     * @param parent     if available the parent of the given target
     */
    GenericEventTarget(String              targetId,
                       Map<String, String> parameters,
                       GenericEventTarget  parent)
    {
        this.targetId = targetId;
        this.parameters = parameters;
        this.parent = parent;
    }

    /**
     * <p>
     * used to add a child to this target
     * </p>
     *
     * @param child the new child of this event target
     */
    synchronized void addChild(GenericEventTarget child) {
        if (children == null) {
            children = new ArrayList<>();
        }
        
        this.children.add(child);
    }

    /**
     * @return the targetId
     */
    String getId() {
        return targetId;
    }

    /**
     * @return the parameters
     */
    Map<String, String> getParameters() {
        return parameters;
    }

    /**
     * @return the parent
     */
    GenericEventTarget getParent() {
        return parent;
    }

    /**
     * @return the children
     */
    List<GenericEventTarget> getChildren() {
        return children;
    }

    /**
     * 
     */
    @Override
    public String toString() {
        return targetId;
    }
}
