//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.commands;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import de.ugoe.cs.autoquest.plugin.html.HTMLLogCompressor;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * This command performs a compression of HTML log files. It traverse a directory structure. All
 * files in the same directory are treated as HTML log files. They are parsed and all GUI elements
 * and events contained in them are dumped into a new log file. No GUI element is dumped more than
 * once. Through this, several log files containing duplicates of GUI elements are condensed to 
 * only one. The events are put at the end of the new log file and are sorted based on their time
 * stamp. 
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDcompressHTMLLogFiles implements Command {

    /* (non-Javadoc)
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String path;

        try {
            path = (String) parameters.get(0);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("illegal parameters provided: " + e);
        }

        File directory = new File(path);
        if (!directory.isDirectory()) {
            Console.printerrln(path + " is not a directory");
            return;
        }

        compressDirectory(directory);

    }

    /**
     * <p>
     * convenience method for recursively calling the {@link HTMLLogCompressor} for each directory
     * </p>
     * 
     * @param the directory to be traversed
     */
    private void compressDirectory(File directory) {
        if (directory.isDirectory()) {
            File[] children = directory.listFiles();
            
        	boolean containsAtLeastOneFile = false;
            if (children != null) {
            	Arrays.sort(children);

            	for (File child : children) {
            		compressDirectory(child);

            		containsAtLeastOneFile |= child.isFile();
            	}
            }
            
            if (containsAtLeastOneFile) {
                HTMLLogCompressor compressor = new HTMLLogCompressor();
                compressor.compressFilesInDirectory(directory);
            }
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "compressHTMLLogFiles <directory>";
    }

}
