//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import static de.ugoe.cs.autoquest.usability.UsabilitySmellDescription.COMMON_TASK_RATE;

import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.usability.UsabilitySmell;

/**
 *
 */
public class CommonTaskRateRuleTest extends AbstractUsabilityEvaluationTC {

    /**
     *
     */
    @Before
    public void setUp() {
        //UsabilitySmellIntensity.defaultCoverageQuantile = 0;
    }
    
    /**
     *
     */
    @Test
    public void testNoTaskCoverage_01() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Interaction elem1 {}" +
            "  Interaction elem2 {}" +
            "}";
        
        // no smell expected, as there are too few recorded actions
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }
    
    /**
     *
     */
    @Test
    public void testNoTaskCoverage_02() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Interaction elem1 {}" +
            "  Interaction elem2 {}" +
            "  Interaction elem3 {}" +
            "  Interaction elem4 {}" +
            "  Interaction elem5 {}" +
            "  Interaction elem6 {}" +
            "  Interaction elem7 {}" +
            "  Interaction elem8 {}" +
            "  Interaction elem9 {}" +
            "  Interaction elem10 {}" +
            "}";
        
        // smell expected, as for each action a different root node exists
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE),
              new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }
    
    /**
     *
     */
    @Test
    public void testBadTaskCoverage_01() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Iteration it1 {" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Iteration it2 {" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Iteration it3 {" +
            "    Interaction elem3 {}" +
            "  }" +
            "  Iteration it4 {" +
            "    Interaction elem4 {}" +
            "  }" +
            "  Iteration it5 {" +
            "    Interaction elem5 {}" +
            "  }" +
            "  Iteration it6 {" +
            "    Interaction elem6 {}" +
            "  }" +
            "  Iteration it7 {" +
            "    Interaction elem7 {}" +
            "  }" +
            "  Iteration it8 {" +
            "    Interaction elem8 {}" +
            "  }" +
            "  Iteration it9 {" +
            "    Interaction elem9 {}" +
            "  }" +
            "  Iteration it10 {" +
            "    Interaction elem10 {}" +
            "  }" +
            "}";
        
        // smell expected, as for each action a different root node exists
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE),
              new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }
    
    /**
     *
     */
    @Test
    public void testBadTaskCoverage_02() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Iteration it3 {" +
            "    Interaction elem3 {}" +
            "  }" +
            "  Iteration it4 {" +
            "    Interaction elem4 {}" +
            "  }" +
            "  Iteration it5 {" +
            "    Interaction elem5 {}" +
            "  }" +
            "  Iteration it6 {" +
            "    Interaction elem6 {}" +
            "  }" +
            "  Iteration it7 {" +
            "    Interaction elem7 {}" +
            "  }" +
            "  Iteration it8 {" +
            "    Interaction elem8 {}" +
            "  }" +
            "  Iteration it9 {" +
            "    Interaction elem9 {}" +
            "  }" +
            "  Iteration it10 {" +
            "    Interaction elem10 {}" +
            "  }" +
            "}";
        
        // smell expected, as for many actions a different root node exists
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE),
              new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }
    
    /**
     *
     */
    @Test
    public void testBadTaskCoverage_03() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Iteration it1 {" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Interaction elem3 {}" +
            "  }" +
            "  Iteration it4 {" +
            "    Interaction elem4 {}" +
            "  }" +
            "  Iteration it5 {" +
            "    Interaction elem5 {}" +
            "  }" +
            "  Iteration it6 {" +
            "    Interaction elem6 {}" +
            "  }" +
            "  Iteration it7 {" +
            "    Interaction elem7 {}" +
            "  }" +
            "  Iteration it8 {" +
            "    Interaction elem8 {}" +
            "  }" +
            "  Iteration it9 {" +
            "    Interaction elem9 {}" +
            "  }" +
            "  Iteration it10 {" +
            "    Interaction elem10 {}" +
            "  }" +
            "}";
        
        // smell expected, as for many actions a different root node exists
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE),
              new UsabilitySmell(UsabilitySmellIntensity.getIntensity(0), COMMON_TASK_RATE) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testPerfectTaskCoverage_01() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Interaction elem3 {}" +
            "    Interaction elem4 {}" +
            "    Interaction elem5 {}" +
            "    Interaction elem6 {}" +
            "    Interaction elem7 {}" +
            "    Interaction elem8 {}" +
            "    Interaction elem9 {}" +
            "    Interaction elem10 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }

    /**
     *
     */
    @Test
    public void testPerfectTaskCoverage_02() {
        CommonTaskRateRule rule = new CommonTaskRateRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Interaction elem3 {}" +
            "    Interaction elem4 {}" +
            "    Interaction elem5 {}" +
            "    Interaction elem6 {}" +
            "    Interaction elem7 {}" +
            "    Interaction elem8 {}" +
            "    Interaction elem9 {}" +
            "    Interaction elem10 {}" +
            "    Interaction elem11 {}" +
            "    Interaction elem12 {}" +
            "    Interaction elem13 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));
    }
    
}
