//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.jfc.commands;

import java.io.File;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.guimodel.GUIModel;
import de.ugoe.cs.autoquest.plugin.jfc.JFCSimplifiedLogParser;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command that tries to parse all files in a folder as if they were log files generated by the
 * JFCMonitor. The result is one set of sequences for all files (not one set of sequences for each
 * file!).
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDparseDirJFC implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String path;
        String sequencesName = "sequences";

        try {
            path = (String) parameters.get(0);
            if (parameters.size() >= 2) {
                sequencesName = (String) parameters.get(1);
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException();
        }

        File folder = new File(path);
        if (!folder.isDirectory()) {
            Console.printerrln(path + " is not a directory");
            return;
        }

        JFCSimplifiedLogParser parser = new JFCSimplifiedLogParser();

        String absolutPath = folder.getAbsolutePath();
        String[] children = folder.list();
        
        if (children != null) {
        	for (String filename : children) {
        		String source = absolutPath + File.separator + filename;
        		Console.traceln(Level.INFO, "Processing file: " + source);

        		try {
        			parser.parseFile(source);
        		}
        		catch (Exception e) {
        			Console.printerrln("Could not parse " + source + ": " + e.getMessage());
        		}
        	}
        }

        Collection<List<Event>> sequences = parser.getSequences();
        
        GUIModel targets = parser.getGuiModel();

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }
        
        if (GlobalDataContainer.getInstance().addData(sequencesName + "_targets", targets)) {
            CommandHelpers.dataOverwritten(sequencesName + "_targets");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "parseDirJFC <directory> {<sequencesName>}";
    }

}
