//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.html.commands;

import java.io.File;
import java.util.List;

import de.ugoe.cs.autoquest.plugin.html.HTMLLogQueryDeleter;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * removes all queries in URL paths from a set of HTML recordings. For this, the command traverses
 * a given directory structure. In each directory it treats the contained files as HTML log files.
 * It parses them, copies all GUI elements and events and but removes query paramters from those
 * GUI elements that contain them. The result is a copy of each treated log file. The origins are
 * deleted. The copies are named as the origins.
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDdeleteHTMLQueries implements Command {

    /* (non-Javadoc)
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String path;
        
        try {
            path = (String) parameters.get(0);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("illegal parameters provided: " + e);
        }

        File directory = new File(path);
        if (!directory.isDirectory()) {
            Console.printerrln(path + " is not a directory");
            return;
        }

        deleteHTMLQueriesInDirectory(directory);

    }

    /**
     * <p>
     * convenience method to traverse the directory structure
     * </p>
     * 
     * @param directory the directory to be treated next
     */
    private void deleteHTMLQueriesInDirectory(File directory) {
        if (directory.isDirectory()) {
            File[] children = directory.listFiles();
            
            if (children != null) {
                for (File child : children) {
                    deleteHTMLQueriesInDirectory(child);
                }
            }
        }
        else if (directory.isFile()) {
            new HTMLLogQueryDeleter().deleteQueries(directory);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "deleteHTMLQueries <directory>";
    }

}
