//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.commands.usage;

import java.util.Collection;
import java.util.List;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.testgeneration.RandomWalkGenerator;
import de.ugoe.cs.autoquest.usageprofiles.IStochasticProcess;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to generate random sessions.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDgenerateRandomSequences implements Command {

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#run(java.util.List)
	 */
	@Override
	public void run(List<Object> parameters) {
		String modelname;
		String sequencesName;
		int numSessions;
		int minLength = 0;
		int maxLength = Integer.MAX_VALUE;
		long maxIter;
		boolean validEnd = true;
		try {
			modelname = (String) parameters.get(0);
			sequencesName = (String) parameters.get(1);
			numSessions = Integer.parseInt((String) parameters.get(2));
			minLength = Integer.parseInt((String) parameters.get(3));
			maxLength = Integer.parseInt((String) parameters.get(4));
			maxIter = numSessions * 10;
			if (parameters.size() >= 5) {
				maxIter = Long.parseLong((String) parameters.get(5));
			}
			if (parameters.size() >= 6) {
				validEnd = Boolean.parseBoolean((String) parameters.get(6));
			}
		} catch (Exception e) {
			throw new IllegalArgumentException();
		}

		IStochasticProcess model = null;
		Object dataObject = GlobalDataContainer.getInstance()
				.getData(modelname);
		if (dataObject == null) {
			CommandHelpers.objectNotFoundMessage(modelname);
			return;
		}
		if (!(dataObject instanceof IStochasticProcess)) {
			CommandHelpers.objectNotType(modelname, "IStochasticProcess");
			return;
		}
		model = (IStochasticProcess) dataObject;

		RandomWalkGenerator generator = new RandomWalkGenerator(numSessions,
				minLength, maxLength, validEnd, maxIter, 100);
		Collection<List<Event>> sequences = generator
				.generateTestSuite(model);

		if (sequences.size() < numSessions) {
			Console.println("Only " + sequences.size()
					+ " unique sessions generated after " + maxIter
					+ " iterations");
		}

		if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
			CommandHelpers.dataOverwritten(sequencesName);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public String help() {
		return "generateRandomSequenecs <modelname> <sequencesName> <numSequences> <minlength> <maxlength> {<maxIter>} {<validEnd>}";
	}

}
