//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.tasktrees.taskequality;

import java.util.Collection;

import de.ugoe.cs.autoquest.eventcore.IEventTarget;
import de.ugoe.cs.autoquest.eventcore.gui.IInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.KeyInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.KeyPressed;
import de.ugoe.cs.autoquest.eventcore.gui.KeyReleased;
import de.ugoe.cs.autoquest.eventcore.gui.KeyTyped;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonDown;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonInteraction;
import de.ugoe.cs.autoquest.eventcore.gui.MouseButtonUp;
import de.ugoe.cs.autoquest.eventcore.gui.MouseClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseDoubleClick;
import de.ugoe.cs.autoquest.eventcore.gui.MouseDragAndDrop;
import de.ugoe.cs.autoquest.eventcore.gui.Scroll;
import de.ugoe.cs.autoquest.eventcore.gui.TextInput;
import de.ugoe.cs.autoquest.eventcore.gui.ValueSelection;
import de.ugoe.cs.autoquest.eventcore.guimodel.IButton;
import de.ugoe.cs.autoquest.eventcore.guimodel.ICheckBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IComboBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IImage;
import de.ugoe.cs.autoquest.eventcore.guimodel.IListBox;
import de.ugoe.cs.autoquest.eventcore.guimodel.IMenu;
import de.ugoe.cs.autoquest.eventcore.guimodel.IMenuButton;
import de.ugoe.cs.autoquest.eventcore.guimodel.IRadioButton;
import de.ugoe.cs.autoquest.eventcore.guimodel.IShape;
import de.ugoe.cs.autoquest.eventcore.guimodel.IText;
import de.ugoe.cs.autoquest.eventcore.guimodel.IToolTip;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.IEventTaskInstance;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITask;
import de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance;

/**
 * <p>
 * This rule compares GUI event tasks (i.e. it is more concrete, than the
 * {@link EventTaskComparisonRule}). Two GUI event tasks are only equal if their event type and
 * target are equal. The returned equality is even more fine-grained for events whose type is
 * {@link TextInput} and {@link ValueSelection}. For text inputs, lexical equality is returned if
 * the same text is entered using the same key interactions. Syntactical equality is returned if
 * the same text is entered using different key interactions. Semantical equality is returned if
 * different text is entered, but into the same event target. Value selections are syntactically
 * equal, if the same value is selected. Otherwise they are semantically equal.
 * </p>
 * 
 * @author Patrick Harms
 */
public class GUIEventTaskComparisonRule implements TaskComparisonRule {
    
    /* (non-Javadoc)
     * @see TaskComparisonRule#isApplicable(ITask, ITask)
     */
    @Override
    public boolean isApplicable(ITask task1, ITask task2) {
        if (!(task1 instanceof IEventTask)) {
            return false;
        }
        
        if (!(task2 instanceof IEventTask)) {
            return false;
        }
        
        IEventTaskInstance instance1 = (IEventTaskInstance) task1.getInstances().iterator().next();
        
        if (!(instance1.getEvent().getType() instanceof IInteraction)) {
            return false;
        }
        
        IEventTaskInstance instance2 = (IEventTaskInstance) task2.getInstances().iterator().next();
        
        if (!(instance2.getEvent().getType() instanceof IInteraction)) {
            return false;
        }
        
        return true;
    }

    /* (non-Javadoc)
     * @see TaskComparisonRule#areLexicallyEqual(ITask, ITask)
     */
    @Override
    public boolean areLexicallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see TaskComparisonRule#areSyntacticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSyntacticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see TaskComparisonRule#areSemanticallyEqual(ITask, ITask)
     */
    @Override
    public boolean areSemanticallyEqual(ITask task1, ITask task2) {
        TaskEquality equality = getEquality(task1, task2, TaskEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see TaskComparisonRule#compare(ITask, ITask)
     */
    @Override
    public TaskEquality compare(ITask task1, ITask task2) {
        return getEquality(task1, task2, null);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.taskequality.TaskComparisonRule#isApplicable(de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance, de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance)
     */
    @Override
    public boolean isApplicable(ITaskInstance instance1, ITaskInstance instance2) {
        return
            (instance1 instanceof IEventTaskInstance) &&
            (instance2 instanceof IEventTaskInstance) &&
            (((IEventTaskInstance) instance1).getEvent().getType() instanceof IInteraction) &&
            (((IEventTaskInstance) instance2).getEvent().getType() instanceof IInteraction);
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.taskequality.TaskComparisonRule#areLexicallyEqual(de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance, de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance)
     */
    @Override
    public boolean areLexicallyEqual(ITaskInstance instance1, ITaskInstance instance2) {
        TaskEquality equality = getEquality(instance1, instance2, TaskEquality.LEXICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.LEXICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.taskequality.TaskComparisonRule#areSyntacticallyEqual(de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance, de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance)
     */
    @Override
    public boolean areSyntacticallyEqual(ITaskInstance instance1, ITaskInstance instance2) {
        TaskEquality equality = getEquality(instance1, instance2, TaskEquality.SYNTACTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.taskequality.TaskComparisonRule#areSemanticallyEqual(de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance, de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance)
     */
    @Override
    public boolean areSemanticallyEqual(ITaskInstance instance1, ITaskInstance instance2) {
        TaskEquality equality = getEquality(instance1, instance2, TaskEquality.SEMANTICALLY_EQUAL);
        return (equality != null) && (equality.isAtLeast(TaskEquality.SEMANTICALLY_EQUAL));
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.tasktrees.taskequality.TaskComparisonRule#compare(de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance, de.ugoe.cs.autoquest.tasktrees.treeifc.ITaskInstance)
     */
    @Override
    public TaskEquality compare(ITaskInstance instance1, ITaskInstance instance2) {
        return getEquality(instance1, instance2, null);
    }

    /**
     * 
     */
    private TaskEquality getEquality(ITask         task1,
                                     ITask         task2,
                                     TaskEquality  requiredEqualityLevel)
    {
        Collection<ITaskInstance> taskInstances1 = task1.getInstances();
        Collection<ITaskInstance> taskInstances2 = task2.getInstances();
        
        if (taskInstances1.size() > taskInstances2.size()) {
            // it may be the case, that all instances of first task match already the first instance
            // of the second task. In this case, the below for loops would iterate of all instances
            // of the first task but the internal loop would always break. But if the first task
            // has many more instances than the second task, performing the comparison the other
            // way around would result in less comparisons. Hence, we switch the task instance lists
            // to have fewer comparisons.
            Collection<ITaskInstance> tmp = taskInstances1;
            taskInstances1 = taskInstances2;
            taskInstances2 = tmp;
        }
        
        TaskEquality checkedEquality =
            requiredEqualityLevel != null ? requiredEqualityLevel : TaskEquality.SEMANTICALLY_EQUAL;
        
        TaskEquality commonDenominator = TaskEquality.LEXICALLY_EQUAL;
        
        for (ITaskInstance instance1 : taskInstances1) {
            TaskEquality mostConcreteEquality = null;
            
            for (ITaskInstance instance2 : taskInstances2) {
                TaskEquality equality = getEquality(instance1, instance2, requiredEqualityLevel);
                
                if ((equality != null) && ((mostConcreteEquality == null) ||
                                           (equality.isAtLeast(mostConcreteEquality))))
                {
                    mostConcreteEquality = equality;
                    
                    if (((requiredEqualityLevel != null) &&
                         (mostConcreteEquality.isAtLeast(requiredEqualityLevel))) ||
                        (mostConcreteEquality.isAtLeast(TaskEquality.LEXICALLY_EQUAL)))
                    {
                        break;
                    }
                }
            }
            
            commonDenominator = commonDenominator.getCommonDenominator(mostConcreteEquality);
            
            if (!commonDenominator.isAtLeast(checkedEquality)) {
                return TaskEquality.UNEQUAL;
            }
        }
        
        return commonDenominator;
    }

    /**
     * 
     */
    private TaskEquality getEquality(ITaskInstance instance1,
                                     ITaskInstance instance2,
                                     TaskEquality  requiredEqualityLevel)
    {
        IEventTaskInstance eventTask1 = (IEventTaskInstance) instance1;
        IEventTaskInstance eventTask2 = (IEventTaskInstance) instance2;
        
        if (!eventTask1.getEvent().getTarget().equals(eventTask2.getEvent().getTarget())) {
            return TaskEquality.UNEQUAL;
        }
        
        IInteraction interaction1 = (IInteraction) eventTask1.getEvent().getType();
        IInteraction interaction2 = (IInteraction) eventTask2.getEvent().getType();
        
        return compareInteractions
            (interaction1, interaction2, eventTask1.getEvent().getTarget(), requiredEqualityLevel);
    }

    /**
     * <p>
     * compares two interactions. The method delegates to other, more specific compare method, e.g.,
     * {@link #compareTextInputs(TextInput, TextInput)} and
     * {@link #compareValueSelections(ValueSelection, ValueSelection)}, if any exist for the
     * concrete interaction types. Otherwise it uses the equals method of the interactions for
     * comparison. In this case, if the interactions equals method returns true, this method
     * returns lexical equality.
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level can be at most as
     * concrete as the provided one. If the provided one is null, it is expected to be lexical
     * equality.
     * </p>
     *
     * @param interaction1  the first interaction to compare
     * @param interaction2  the second interaction to compare
     * @param eventTarget   the event target on which the interactions happened (used within
     *                      special comparisons like mouse clicks on buttons, where the coordinates
     *                      can be ignored)
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareInteractions(IInteraction interaction1,
                                             IInteraction interaction2,
                                             IEventTarget eventTarget,
                                             TaskEquality equalityLevel)
    {
        TaskEquality level = equalityLevel;
        
        if (level == null) {
            level = TaskEquality.LEXICALLY_EQUAL;
        }
        
        if (interaction1 == interaction2) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else if ((interaction1 instanceof KeyInteraction) &&
                 (interaction2 instanceof KeyInteraction))
        {
            return compareKeyInteractions
                ((KeyInteraction) interaction1, (KeyInteraction) interaction2, level);
        }
        else if ((interaction1 instanceof MouseButtonInteraction) &&
                 (interaction2 instanceof MouseButtonInteraction))
        {
            return compareMouseButtonInteractions
                ((MouseButtonInteraction) interaction1, (MouseButtonInteraction) interaction2,
                 eventTarget, level);
        }
        else if ((interaction1 instanceof Scroll) && (interaction2 instanceof Scroll)) {
            return compareScrolls((Scroll) interaction1, (Scroll) interaction2, level);
        }
        else if ((interaction1 instanceof TextInput) && (interaction2 instanceof TextInput)) {
            return compareTextInputs
                ((TextInput) interaction1, (TextInput) interaction2, level);
        }
        else if ((interaction1 instanceof ValueSelection) &&
                 (interaction2 instanceof ValueSelection))
        {
            return compareValueSelections
                ((ValueSelection<?>) interaction1, (ValueSelection<?>) interaction2, level);
        }
        else if (interaction1.equals(interaction2)) {
            return TaskEquality.LEXICALLY_EQUAL;
        }
        else {
            return TaskEquality.UNEQUAL;
        }
    }

    /**
     * <p>
     * compares two key interactions. If both are of the same type and if both have the
     * same key, they are lexically equal. If both are only of the same type, they are
     * semantically equal. Otherwise, they are unequal.
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level is as concrete as
     * the provided one. It may be more concrete if there is no difference regarding the
     * comparison on the levels.
     * </p>
     *
     * @param interaction1  the first key interaction
     * @param interaction2  the second key interaction
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareKeyInteractions(KeyInteraction interaction1,
                                                KeyInteraction interaction2,
                                                TaskEquality   equalityLevel)
    {
        if (((interaction1 instanceof KeyPressed) && (interaction2 instanceof KeyPressed)) ||
            ((interaction1 instanceof KeyReleased) && (interaction2 instanceof KeyReleased)) ||
            ((interaction1 instanceof KeyTyped) && (interaction2 instanceof KeyTyped)))
        {
            if ((equalityLevel.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL)) &&
                (interaction1.getKey() == interaction2.getKey()))
            {
                return TaskEquality.LEXICALLY_EQUAL;
            }
            else {
                // pressing a key on the same target, e.g. a text field, usually has the same
                // semantics
                return TaskEquality.SEMANTICALLY_EQUAL;
            }
        }
        
        return TaskEquality.UNEQUAL;
    }
    
    /**
     * <p>
     * compares two mouse drag and drops. If both drag and drops have the same start and end
     * coordinates, they are lexically equal. Otherwise, they are semantically equal.
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level is as concrete as
     * the provided one. It may be more concrete if there is no difference regarding the
     * comparison on the levels.
     * </p>
     *
     * @param interaction1  the first mouse drag and drop to compare
     * @param interaction2  the second mouse drag and drop to compare
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareMouseDragAndDrops(MouseDragAndDrop interaction1,
                                                  MouseDragAndDrop interaction2,
                                                  TaskEquality     equalityLevel)
    {
        if (interaction1.getButton() != interaction2.getButton()) {
            return TaskEquality.UNEQUAL;
        }
        
        if (equalityLevel.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL)) {
            int x1 = interaction1.getX();
            int x1Start = interaction1.getXStart();
            int x2 = interaction2.getX();
            int x2Start = interaction2.getXStart();
            int y1 = interaction1.getY();
            int y1Start = interaction1.getYStart();
            int y2 = interaction2.getY();
            int y2Start = interaction2.getYStart();
        
            if ((x1Start == x2Start) && (x1 == x2) && (y1Start == y2Start) && (y1 == y2)) {
                return TaskEquality.LEXICALLY_EQUAL;
            }
        }
        
        // performing drag and drops on the same target usually have the same semantic meaning,
        // i.e., the same function is called
        return TaskEquality.SEMANTICALLY_EQUAL;
    }

    /**
     * <p>
     * compares two mouse button interactions such as clicks, mouse button down, or double clicks.
     * If both interactions have the same coordinates, they are lexically equal. Otherwise, they
     * are semantically equal. Mouse clicks for which the coordinates make no lexical difference
     * (see {@link #clickCoordinatesMakeLexicalDifference(IEventTarget)}) are treated as
     * lexically equal.
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level is as concrete as
     * the provided one. It may be more concrete if there is no difference regarding the
     * comparison on the levels.
     * </p>
     *
     * @param interaction1  the first mouse button interaction to compare
     * @param interaction2  the second mouse button interaction to compare
     * @param eventTarget   the event target on which the interactions happened (used within
     *                      special comparisons like mouse clicks on buttons, where the coordinates
     *                      can be ignored)
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareMouseButtonInteractions(MouseButtonInteraction interaction1,
                                                        MouseButtonInteraction interaction2,
                                                        IEventTarget           eventTarget,
                                                        TaskEquality           equalityLevel)
    {
        boolean coordinatesMatch = true;
        
        if ((interaction1 instanceof MouseDragAndDrop) &&
            (interaction2 instanceof MouseDragAndDrop))
        {
            return compareMouseDragAndDrops
                ((MouseDragAndDrop) interaction1, (MouseDragAndDrop) interaction2, equalityLevel);
        }
        else if (interaction1.getButton() != interaction2.getButton()) {
            return TaskEquality.UNEQUAL;
        }
        else if (equalityLevel.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL) &&
                 clickCoordinatesMakeLexicalDifference(eventTarget))
        {
            int x1 = interaction1.getX();
            int x2 = interaction2.getX();
            int y1 = interaction1.getY();
            int y2 = interaction2.getY();

            if ((x1 != x2) || (y1 != y2)) {
                coordinatesMatch = false;
            }
        }
        
        // up to now, they can be equal. Now check the types. Do it as last action as these
        // checks take the most time and should, therefore, only be done latest
        if (((interaction1 instanceof MouseClick) && (interaction2 instanceof MouseClick)) ||
            ((interaction1 instanceof MouseDoubleClick) &&
             (interaction2 instanceof MouseDoubleClick)) ||
            ((interaction1 instanceof MouseButtonDown) &&
             (interaction2 instanceof MouseButtonDown)) ||
            ((interaction1 instanceof MouseButtonUp) &&
             (interaction2 instanceof MouseButtonUp)))
        {
            if (coordinatesMatch) {
                return TaskEquality.LEXICALLY_EQUAL;
            }
            else {
                // in most situations, the coordinates are not of interest. But if they are, then
                // the event can be at most semantically equal
                return TaskEquality.SEMANTICALLY_EQUAL;
            }
        }
        
        return TaskEquality.UNEQUAL;
    }

    /**
     * <p>
     * compares two scrolls and considers them as lexically equal if they have the same coordinates.
     * Otherwise, they are syntactically equal as the happen on the same target
     * </p>
     *
     * @param interaction1  the first scroll interaction to compare
     * @param interaction2  the second scroll interaction to compare
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareScrolls(Scroll       interaction1,
                                        Scroll       interaction2,
                                        TaskEquality equalityLevel)
    {
        if (equalityLevel.isAtLeast(TaskEquality.LEXICALLY_EQUAL)) {
            int x1 = interaction1.getXPosition();
            int x2 = interaction2.getXPosition();
            int y1 = interaction1.getYPosition();
            int y2 = interaction2.getYPosition();
        
            if ((x1 == x2) && (y1 == y2)) {
                return TaskEquality.LEXICALLY_EQUAL;
            }
        }
        
        return TaskEquality.SYNTACTICALLY_EQUAL;
    }

    /**
     * <p>
     * compares two text inputs. If both text inputs have the same entered text and text input
     * events, they are lexically equal. If they only have the same entered text, they are
     * syntactically equal. If they are only both text inputs, they are semantically equal.
     * (the equality of the event targets is checked beforehand).
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level is as concrete as
     * the provided one. It may be more concrete if there is no difference regarding the
     * comparison on the levels.
     * </p>
     *
     * @param interaction1  the first text input to compare
     * @param interaction2  the second text input to compare
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareTextInputs(TextInput    interaction1,
                                           TextInput    interaction2,
                                           TaskEquality equalityLevel)
    {
        switch (equalityLevel) {
            case LEXICALLY_EQUAL:
                if (interaction1.getTextInputEvents().equals(interaction2.getTextInputEvents())) {
                    return TaskEquality.LEXICALLY_EQUAL;
                }
                // fall through
            case SYNTACTICALLY_EQUAL:
                if (interaction1.getEnteredText().equals(interaction2.getEnteredText())) {
                    return TaskEquality.SYNTACTICALLY_EQUAL;
                }
                // fall through
            case SEMANTICALLY_EQUAL:
                return TaskEquality.SEMANTICALLY_EQUAL;
            default:
                return TaskEquality.UNEQUAL;
        }
    }

    /**
     * <p>
     * compares two value selections. If both value selections have the same selected value, they
     * are syntactically equal, otherwise they are semantically equal.
     * (the equality of the event targets is checked beforehand).
     * </p>
     * <p>
     * The provided equality level can be used to restrict the quality check to the given level.
     * This is done for optimization purposes. The returned equality level is as concrete as
     * the provided one. It may be more concrete if there is no difference regarding the
     * comparison on the levels.
     * </p>
     *
     * @param interaction1  the first value selection to compare
     * @param interaction2  the second value selection to compare
     * @param equalityLevel the equality level to be checked for
     * 
     * @return as described
     */
    private TaskEquality compareValueSelections(ValueSelection<?> interaction1,
                                                ValueSelection<?> interaction2,
                                                TaskEquality      equalityLevel)
    {
        if (equalityLevel.isAtLeast(TaskEquality.SYNTACTICALLY_EQUAL)) {
            Object value1 = interaction1.getSelectedValue();
            Object value2 = interaction2.getSelectedValue();
        
            if ((value1 == value2) || ((value1 != null) && (value1.equals(value2)))) {
                return TaskEquality.LEXICALLY_EQUAL;
            }
        }
        
        return TaskEquality.SEMANTICALLY_EQUAL;
    }

    /**
     * <p>
     * Checks, if the coordinates of a click or double click on the provided event target makes
     * a lexical difference. Mouse clicks and double clicks on buttons, check boxes,
     * combo boxes, images, list boxes, menu buttons, radio buttons, shapes, uneditable text,
     * and tool tips have no lexical difference as long as they happen on the same event target.
     * The concrete coordinates are not relevant.
     * </p>
     *
     * @param eventTarget the event target on which the interaction occurred
     * 
     * @return if the coordinates are important to be considered for clicks and double clicks,
     *         false else
     */
    private boolean clickCoordinatesMakeLexicalDifference(IEventTarget eventTarget) {
        if ((eventTarget instanceof IButton) ||
            (eventTarget instanceof ICheckBox) ||
            (eventTarget instanceof IComboBox) ||
            (eventTarget instanceof IImage) ||
            (eventTarget instanceof IListBox) ||
            (eventTarget instanceof IMenu) ||
            (eventTarget instanceof IMenuButton) ||
            (eventTarget instanceof IRadioButton) ||
            (eventTarget instanceof IShape) ||
            (eventTarget instanceof IText) ||
            (eventTarget instanceof IToolTip))
        {
            return false;
        }
        else {
            return true;
        }
    }

}
