//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.genericeventmonitor;

import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * The generic event monitor starts a web server ({@link GenericEventMonitorServer}) that receives
 * events in JSON format. These events are logged using the {@link GenericEventMonitorLogManager}.
 * The class assures that on shutdown e.g. caused by CTRL-C the server and the log manager are
 * stopped correctly.
 * </p>
 * 
 * @author Patrick Harms
 */
public class GenericEventMonitor implements GenericEventMonitorComponent {

    /**
     * the port on which the webserver shall listen. Defaults to 8090.
     */
    private int port = 8090;
    
    /**
     * the web server receiving the log messages
     */
    private GenericEventMonitorServer server;
    
    /**
     * the directory into which the log files shall be written
     */
    private String logFileBaseDir;

    /**
     * the log manager being responsible for performing the logging
     */
    private GenericEventMonitorLogManager logManager;

    /**
     * the thread needed to handle CTRL-C events
     */
    private Thread shutdownHook;

    /**
     * <p>
     * initializes the monitor with the command line arguments. Those may be the log directory
     * as first argument and the port to listen on as second
     * </p>
     *
     * @param commandLineArguments the command line arguments when starting the monitor using
     *                             the {@link Runner}
     */
    public GenericEventMonitor(String[] commandLineArguments) {
        if (commandLineArguments.length > 0) {
            this.logFileBaseDir = commandLineArguments[0];
            Console.println("putting logs into directory " + this.logFileBaseDir);
        }
        
        if (commandLineArguments.length > 1) {
            try {
                this.port = Integer.parseInt(commandLineArguments[1]);
            }
            catch (NumberFormatException e) {
                Console.println("ignoring invalid port specification " + commandLineArguments[1]);
            }
            Console.println("listening on port " + this.port);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.genericeventmonitor.GenericEventMonitorComponent#init()
     */
    @Override
    public synchronized void init() throws GenericEventMonitorException {
        if (server != null) {
            throw new IllegalStateException("already initialized.");
        }
        
        try {
            logManager = new GenericEventMonitorLogManager(logFileBaseDir);
            logManager.init();
        
            server = new GenericEventMonitorServer(port, logManager);
            server.init();
        
            shutdownHook = new Thread(new ShutdownHook(server, logManager));
        }
        catch (GenericEventMonitorException e) {
            Console.printerrln("could not initialize generic event monitor: " + e);
            Console.logException(e);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.genericeventmonitor.GenericEventMonitorComponent#start()
     */
    @Override
    public synchronized void start() {
        if (server == null) {
            throw new IllegalStateException("not initialized.");
        }
        
        try {
            Runtime.getRuntime().addShutdownHook(shutdownHook);
            logManager.start();
            server.start();
        }
        catch (GenericEventMonitorException e) {
            Console.printerrln("could not start generic event monitor: " + e);
            Console.logException(e);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.genericeventmonitor.GenericEventMonitorComponent#stop()
     */
    @Override
    public synchronized void stop() {
        if (server == null) {
            throw new IllegalStateException("not initialized.");
        }
        
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
        server.stop();
        logManager.stop();
        
        server = null;
        logManager = null;
    }

}
