//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.usability;

import static de.ugoe.cs.autoquest.usability.UsabilitySmellDescription.INEFFICIENT_ACTIONS;

import org.junit.Before;
import org.junit.Test;

import de.ugoe.cs.autoquest.usability.UsabilitySmell;

/**
 *
 */
public class RequiredInefficientActionsRuleTest extends AbstractUsabilityEvaluationTC {

    /**
     *
     */
    @Before
    public void setUp() {
        //UsabilitySmellIntensity.defaultCoverageQuantile = 0;
    }
    
    /**
     *
     */
    @Test
    public void testWithNormalScroll_01() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "}";
        
        // no smell expected, as interactions do not form tasks
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] {  };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_02() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "  Scroll body {}" +
            "  Interaction elem1 {}" +
            "}";
        
        // no smell expected, as interactions do not form tasks
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_03() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_04() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Scroll body {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_05() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testWithNormalScroll_06() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*HIGH*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_01() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    Interaction elem1 {}" +
            "    Scroll body {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_02() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_03() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*MEDIUM*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_04() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Optional opt1 {}" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem2 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
           { new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_05() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
           { new UsabilitySmell(/*LOW*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_06() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
           "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_07() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "    Interaction elem2 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem2 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[]
            { new UsabilitySmell(/*INFO*/ UsabilitySmellIntensity.getIntensity(0),
                                  INEFFICIENT_ACTIONS) };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_08() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "  Sequence seq1 {" +
            "    Optional opt1 { }" +
            "    Interaction elem1 {}" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }

    /**
     *
     */
    @Test
    public void testScrollSeldomRequired_09() {
        RequiredInefficientActionsRule rule = new RequiredInefficientActionsRule();

        // ===== check =====
        String spec =
            "UserSession {" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 {" +
            "      Scroll body {}" +
            "    }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "  Sequence seq1 {" +
            "    Interaction elem1 {}" +
            "    Optional opt1 { }" +
            "  }" +
            "}";
        
        UsabilitySmell[] expectedSmells = new UsabilitySmell[] { };

        assertUsabilityEvaluationResult(expectedSmells, rule.evaluate(createTaskModel(spec)));

    }
}
