//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.eventcore.gui;

/**
 * <p>
 * Event type for a mouse drag and drop, i.e., pressing the left mouse button on an element,
 * moving the mouse and releasing the mouse button afterwards.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public class MouseDragAndDrop extends MouseButtonInteraction {

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;
    
    /**
     * <p>
     * the x coordinate, where the drag started
     * </p>
     */
    private int xStart;
    
    /**
     * <p>
     * the y coordinate, where the drag started
     * </p>
     */
    private int yStart;

    /**
     * <p>
     * Constructor. Creates a new {@link MouseDragAndDrop} event type with the left mouse button
     * as button per default
     * </p>
     * 
     * @param xStart the x coordinate, where the drag started
     * @param yStart the y coordinate, where the drag started
     * @param xEnd   the x coordinate, where the drop ended
     * @param yEnd   the y coordinate, where the drop ended
     * 
     * 
     * @see MouseButtonInteraction#MouseButtonInteraction(Button,int,int)
     */
    public MouseDragAndDrop(int xStart, int yStart, int xEnd, int yEnd) {
        super(Button.LEFT, xEnd, yEnd);
        this.xStart = xStart;
        this.yStart = yStart;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.harms.attef.userinteraction.Interaction#getName()
     */
    public String getName() {
        return "MouseDragAndDrop";
    }

    /**
     * @return the x coordinate, where the drag started
     */
    public int getXStart() {
        return xStart;
    }

    /**
     * @return the y coordinate, where the drag started
     */
    public int getYStart() {
        return yStart;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return "mouse drag and drop ((" +
            xStart + "," + yStart + ")-->(" + getX() + "," + getY() + "))";
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof MouseDragAndDrop) {
            return
                (getX() == ((MouseDragAndDrop) obj).getX()) &&
                (getY() == ((MouseDragAndDrop) obj).getY()) &&
                (getXStart() == ((MouseDragAndDrop) obj).getXStart()) &&
                (getYStart() == ((MouseDragAndDrop) obj).getYStart());
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return getX() + getY() + getXStart() + getYStart();
    }
}
