//   Copyright 2012 Georg-August-Universität Göttingen, Germany
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.

package de.ugoe.cs.autoquest.plugin.genericevents.commands;

import java.io.File;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import de.ugoe.cs.autoquest.CommandHelpers;
import de.ugoe.cs.autoquest.eventcore.Event;
import de.ugoe.cs.autoquest.eventcore.IHierarchicalEventTargetModel;
import de.ugoe.cs.autoquest.plugin.genericevents.GenericEventLogParser;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command that tries to parse all files in a folder as if they were log files generated by the
 * generic event monitor. The result is one set of sequences for all files (not one set of
 * sequences for each file!).
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDparseDirGenericEvents implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @Override
    public void run(List<Object> parameters) {
        String path = null;
        String sequencesName = null;
        
        Set<String> ignoredEvents = new HashSet<>();
        
        try {
            for (int i = 0; i < parameters.size(); i++) {
                String param = (String) parameters.get(i);
                if (path == null) {
                    path = param;
                }
                else if (sequencesName == null) {
                    sequencesName = param;
                }
                else {
                    ignoredEvents.add(param);
                }
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("illegal parameters provided: " + e);
        }
        
        if (sequencesName == null) {
            sequencesName = "sequences";
        }

        File folder = new File(path);
        if (!folder.isDirectory()) {
            Console.printerrln(path + " is not a directory");
            return;
        }

        GenericEventLogParser parser = new GenericEventLogParser(ignoredEvents);

        parseFile(folder, parser);

        Collection<List<Event>> sequences = parser.getSequences();

        IHierarchicalEventTargetModel<?> targets = parser.getHierarchicalEventTargetModel();

        if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }

        if (GlobalDataContainer.getInstance().addData(sequencesName + "_targets", targets)) {
            CommandHelpers.dataOverwritten(sequencesName + "_targets");
        }
    }

    /**
     * <p>
     * recursive method for parsing a directory structures
     * </p>
     *
     * @param file   the file object to be parsed. If the file is a folder, the method calls itself
     *               for all children
     * @param parser the parser to use for parsing the files.
     */
    private void parseFile(File file, GenericEventLogParser parser) {
        if (file.isDirectory()) {
            String[] children = file.list();
            
            if (children != null) {
            	Arrays.sort(children);

            	for (String child : children) {
            		File childFile = new File(file, child);
            		parseFile(childFile, parser);
            	}
            }
        }
        else if (file.isFile()) {
            String source = file.getAbsolutePath();
            Console.traceln(Level.INFO, "Processing file: " + source);

            try {
                parser.parseFile(file);
            }
            catch (Exception e) {
                Console.printerrln("Could not parse " + source + ": " + e.getMessage());
                e.printStackTrace();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "parseDirGenericEvents <directory> [<sequencesName>] [<ignoredEventType>*] [<ignoredEventType.ignoredEventTargetId>*]";
    }

}
