package de.ugoe.cs.eventbensch.jfc;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.security.InvalidParameterException;
import java.util.Collection;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import de.ugoe.cs.eventbensch.jfc.data.JFCEvent;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * This class provides functionality to parse XML log files generated by the
 * JFCMonitor of EventBench. The result of parsing a file is a collection of
 * event sequences.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class JFCLogParser extends DefaultHandler {

	/**
	 * <p>
	 * Collection of event sequences that is contained in the log file, which is
	 * parsed.
	 * </p>
	 */
	Collection<List<JFCEvent>> sequences;

	/**
	 * <p>
	 * Returns the collection of event sequences that is obtained from parsing
	 * log files.
	 * </p>
	 * 
	 * @return collection of event sequences
	 */
	public Collection<List<JFCEvent>> getSequences() {
		return sequences;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String,
	 * java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName,
			Attributes atts) throws SAXException {
		if (qName.equals("newsession")) {
			Console.traceln("start of session");
			// TODO implement handling: newsession node
		} else if (qName.equals("event")) {
			// TODO implement handling: start of event node
		} else if (qName.equals("param")) {
			// TODO implement handling: param node
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String,
	 * java.lang.String, java.lang.String)
	 */
	@Override
	public void endElement(String uri, String localName, String qName)
			throws SAXException {
		if (qName.equals("event")) {
			// TODO implement handling: end of event node
		} else if (qName.equals("source")) {
			// TODO implement handling: end of source node
		} else if (qName.equals("parent")) {
			// TODO implement handling: end of parent node
		}
	}

	/**
	 * <p>
	 * Parses a log file written by the JFCMonitor and creates a collection of
	 * event sequences.
	 * </p>
	 * 
	 * @param filename
	 *            name and path of the log file
	 */
	public void parseFile(String filename) {
		if (filename == null) {
			throw new InvalidParameterException("filename must not be null");
		}

		SAXParserFactory spf = SAXParserFactory.newInstance();
		spf.setValidating(true);

		SAXParser saxParser = null;
		InputSource inputSource = null;
		try {
			saxParser = spf.newSAXParser();
			inputSource = new InputSource(new InputStreamReader(
					new FileInputStream(filename), "UTF-16"));
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		if (inputSource != null) {
			inputSource.setSystemId("file://"
					+ new File(filename).getAbsolutePath());
			try {
				if (saxParser == null) {
					throw new RuntimeException("SAXParser creation failed");
				}
				saxParser.parse(inputSource, this);
			} catch (SAXParseException e) {
				Console.printerrln("Failure parsing file in line "
						+ e.getLineNumber() + ", column " + e.getColumnNumber()
						+ ".");
				e.printStackTrace();
			} catch (SAXException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

}
