package de.ugoe.cs.eventbench.commands;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;

import de.ugoe.cs.eventbench.CommandHelpers;
import de.ugoe.cs.eventbench.data.Event;
import de.ugoe.cs.eventbench.data.GlobalDataContainer;
import de.ugoe.cs.eventbench.models.IStochasticProcess;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * Command to generate all sequences of a given length.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDgenerateFixedLengthSequences implements Command {

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#run(java.util.List)
	 */
	@Override
	public void run(List<Object> parameters) {
		String modelname;
		String sequencesName;
		int minLength;
		int maxLength;
		boolean all = true;
		int numSequences = -1;
		boolean validEnd = true;
		try {
			modelname = (String) parameters.get(0);
			sequencesName = (String) parameters.get(1);
			minLength = Integer.parseInt((String) parameters.get(2));
			maxLength = Integer.parseInt((String) parameters.get(3));
			if (parameters.size() >= 5) {
				all = Boolean.parseBoolean((String) parameters.get(4));
			}
			if (parameters.size() >= 6) {
				numSequences = Integer.parseInt((String) parameters.get(5));
			}
			if (parameters.size() >= 7) {
				validEnd = Boolean.parseBoolean((String) parameters.get(6));
			}
		} catch (Exception e) {
			throw new InvalidParameterException();
		}

		IStochasticProcess model = null;
		Object dataObject = GlobalDataContainer.getInstance()
				.getData(modelname);
		if (dataObject == null) {
			CommandHelpers.objectNotFoundMessage(modelname);
			return;
		} else if (!(dataObject instanceof IStochasticProcess)) {
			CommandHelpers.objectNotType(modelname, "IStochasticProcess");
			return;
		}
		model = (IStochasticProcess) dataObject;
		Collection<List<? extends Event<?>>> sequences = new LinkedHashSet<List<? extends Event<?>>>();
		for (int length = minLength; length <= maxLength; length++) {
			if( validEnd ) {
				sequences.addAll(model.generateValidSequences(length + 2));
			} else {
				sequences.addAll(model.generateSequences(length + 1, true));
			}
		}
		Console.traceln("" + sequences.size() + " possible");
		if (!all && numSequences < sequences.size()) {
			List<Double> probabilities = new ArrayList<Double>(sequences.size());
			double probSum = 0.0;
			for (List<? extends Event<?>> sequence : sequences) {
				double prob = model.getProbability(sequence);
				probabilities.add(prob);
				probSum += prob;
			}
			Set<Integer> drawnSequences = new HashSet<Integer>(numSequences);
			Random r = new Random();
			while (drawnSequences.size() < numSequences) {
				double randVal = r.nextDouble() * probSum;
				double sum = 0.0d;
				int index = -1;
				while (sum < randVal) {
					index++;
					double currentProb = probabilities.get(index);
					sum += currentProb;
				}
				if (!drawnSequences.contains(index)) {
					drawnSequences.add(index);
					probSum -= probabilities.get(index);
					probabilities.set(index, 0.0d);
				}
			}
			Collection<List<? extends Event<?>>> retainedSequences = new LinkedList<List<? extends Event<?>>>();
			int index = 0;
			for (List<? extends Event<?>> sequence : sequences) {
				if (drawnSequences.contains(index)) {
					retainedSequences.add(sequence);
				}
				index++;
			}
			sequences = retainedSequences;
		}
		if (GlobalDataContainer.getInstance().addData(sequencesName, sequences)) {
			CommandHelpers.dataOverwritten(sequencesName);
		}
		Console.println("" + sequences.size() + " sequences generated");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public void help() {
		Console.println("Usage: generateFixedLengthSequences <modelname> <sequencesName> <minlenght> <maxlength> {<all>} {<numSequences>} {<validEnd>}");
	}

}
