package de.ugoe.cs.quest.usageprofiles;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import de.ugoe.cs.quest.eventcore.Event;
import de.ugoe.cs.quest.usageprofiles.PredictionByPartialMatch;

import java.util.Random;
import org.junit.*;

import static org.junit.Assert.*;

/**
 * The class <code>PredictionByPartialMatchTest</code> contains tests for the
 * class <code>{@link PredictionByPartialMatch}</code>.
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class PredictionByPartialMatchTest {

	Collection<List<? extends Event<?>>> sequences;

	@Test
	public void testPredictionByPartialMatch_1() throws Exception {
		int markovOrder = 2;
		Random r = new Random();

		PredictionByPartialMatch result = new PredictionByPartialMatch(
				markovOrder, r);

		assertNotNull(result);
		assertEquals(markovOrder+1, result.trieOrder);
		assertEquals(0, result.minOrder);
		assertEquals(r, result.r);
		assertEquals(0.1, result.probEscape, 0.0001);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_2() throws Exception {
		int markovOrder = -1;
		Random r = new Random();

		new PredictionByPartialMatch(markovOrder, r);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_3() throws Exception {
		int markovOrder = 2;
		Random r = null;

		new PredictionByPartialMatch(markovOrder, r);
	}
	
	@Test
	public void testPredictionByPartialMatch_4() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;

		PredictionByPartialMatch result = new PredictionByPartialMatch(
				markovOrder, r, probEscape);

		assertNotNull(result);
		assertEquals(markovOrder+1, result.trieOrder);
		assertEquals(0, result.minOrder);
		assertEquals(r, result.r);
		assertEquals(probEscape, result.probEscape, 0.0001);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_5() throws Exception {
		int markovOrder = -1;
		Random r = new Random();
		double probEscape = 0.2;

		new PredictionByPartialMatch(markovOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_6() throws Exception {
		int markovOrder = 2;
		Random r = null;
		double probEscape = 0.2;

		new PredictionByPartialMatch(markovOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_7() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.0;

		new PredictionByPartialMatch(markovOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_8() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 1.0;

		new PredictionByPartialMatch(markovOrder, r, probEscape);
	}
	
	@Test
	public void testPredictionByPartialMatch_9() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;

		PredictionByPartialMatch result = new PredictionByPartialMatch(
				markovOrder, minOrder, r, probEscape);

		assertNotNull(result);
		assertEquals(markovOrder+1, result.trieOrder);
		assertEquals(minOrder, result.minOrder);
		assertEquals(r, result.r);
		assertEquals(probEscape, result.probEscape, 0.0001);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_10() throws Exception {
		int markovOrder = -1;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_11() throws Exception {
		int markovOrder = 2;
		Random r = null;
		double probEscape = 0.2;
		int minOrder = 1;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_12() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.0;
		int minOrder = 1;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_13() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 1.0;
		int minOrder = 1;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_14() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 3;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}
	
	@Test(expected = java.security.InvalidParameterException.class)
	public void testPredictionByPartialMatch_15() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = -1;

		new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
	}

	@Test
	public void testGetProbEscape_1() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.probEscape = probEscape;
		
		double result = fixture.getProbEscape();

		assertEquals(probEscape, result, 0.0001);
	}

	@Test
	public void testGetProbability_1() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("b");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(0.88d, result, 0.0001);
	}
	
	@Test
	public void testGetProbability_2() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("c");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(0.04d, result, 0.0001);
	}
	
	@Test
	public void testGetProbability_3() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 2;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("b");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(1.0d, result, 0.0001);
	}
	
	@Test
	public void testGetProbability_4() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 2;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("c");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(0.0d, result, 0.0001);
	}
	
	@Test
	public void testGetProbability_5() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 0;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("b");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(0.8701d, result, 0.0001);
	}
	
	@Test
	public void testGetProbability_6() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 0;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
		fixture.train(sequences);
		
		List<Event<?>> context = new ArrayList<Event<?>>();
		context.add(Event.STARTEVENT);
		context.add(new Event<String>("a"));

		Event<String> symbol = new Event<String>("c");
		
		double result = fixture.getProbability(context, symbol);
		
		assertEquals(0.0350, result, 0.0001);
	}

	@Test
	public void testSetProbEscape_1() throws Exception {
		int markovOrder = 2;
		Random r = new Random();
		double probEscape = 0.2;
		int minOrder = 1;
		double newProbEscape = 0.3;

		PredictionByPartialMatch fixture = new PredictionByPartialMatch(markovOrder, minOrder, r, probEscape);
				
		fixture.setProbEscape(newProbEscape);

		assertEquals(newProbEscape, fixture.probEscape, 0.0001);
	}

	@Before
	public void setUp() throws Exception {
		List<Event<?>> sequence = new ArrayList<Event<?>>();
		sequence.add(new Event<String>("a"));
		sequence.add(new Event<String>("b"));
		sequence.add(new Event<String>("r"));
		sequence.add(new Event<String>("a"));
		sequence.add(new Event<String>("c"));
		sequence.add(new Event<String>("a"));
		sequence.add(new Event<String>("d"));
		sequence.add(new Event<String>("a"));
		sequence.add(new Event<String>("b"));
		sequence.add(new Event<String>("r"));
		sequence.add(new Event<String>("a"));

		sequences = new ArrayList<List<? extends Event<?>>>();
		sequences.add(sequence);
	}

	public static void main(String[] args) {
		new org.junit.runner.JUnitCore()
				.run(PredictionByPartialMatchTest.class);
	}
}