
package de.ugoe.cs.quest.usageprofiles;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import org.junit.*;

import de.ugoe.cs.quest.eventcore.Event;
import de.ugoe.cs.quest.usageprofiles.FirstOrderMarkovModel;
import de.ugoe.cs.quest.usageprofiles.HighOrderMarkovModel;
import de.ugoe.cs.quest.usageprofiles.ModelFlattener;
import de.ugoe.cs.quest.usageprofiles.PredictionByPartialMatch;
import de.ugoe.cs.quest.usageprofiles.TrieNode;
import de.ugoe.cs.quest.usageprofiles.mock.StringEventType;
import static org.junit.Assert.*;

/**
 * The class <code>ModelFlattenerTest</code> contains tests for the class
 * <code>{@link ModelFlattener}</code>.
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class ModelFlattenerTest {

    List<Event> sequence;

    private static void assertCollectionContent(Collection<?> c1, Collection<?> c2) {
        assertEquals(c1.size(), c2.size());
        for (Object obj : c1) {
            assertTrue(c2.contains(obj));
        }
    }

    @Test
    public void testFlattenHighOrderMarkovModel_1() throws Exception {
        ModelFlattener fixture = new ModelFlattener();
        HighOrderMarkovModel model = new HighOrderMarkovModel(2, new Random());
        Collection<List<Event>> sequences = new ArrayList<List<Event>>();
        sequences.add(sequence);
        model.train(sequences);

        Collection<Event> expectedSymbols = new HashSet<Event>();
        expectedSymbols.add(new Event(new StringEventType("a-=-END")));
        expectedSymbols.add(new Event(new StringEventType("a-=-b")));
        expectedSymbols.add(new Event(new StringEventType("a-=-c")));
        expectedSymbols.add(new Event(new StringEventType("a-=-d")));
        expectedSymbols.add(new Event(new StringEventType("b-=-r")));
        expectedSymbols.add(new Event(new StringEventType("c-=-a")));
        expectedSymbols.add(new Event(new StringEventType("d-=-a")));
        expectedSymbols.add(new Event(new StringEventType("r-=-a")));
        expectedSymbols.add(new Event(new StringEventType("START-=-a")));

        FirstOrderMarkovModel result = fixture.flattenHighOrderMarkovModel(model);

        assertCollectionContent(expectedSymbols, result.getEvents());

        TrieNode<Event> root = result.trie.find(null);
        TrieNode<Event> root_aEnd = root.getChild(new Event(new StringEventType("a-=-END")));
        TrieNode<Event> root_ab = root.getChild(new Event(new StringEventType("a-=-b")));
        TrieNode<Event> root_ab_br = root_ab.getChild(new Event(new StringEventType("b-=-r")));
        TrieNode<Event> root_ac = root.getChild(new Event(new StringEventType("a-=-c")));
        TrieNode<Event> root_ac_ca = root_ac.getChild(new Event(new StringEventType("c-=-a")));
        TrieNode<Event> root_ad = root.getChild(new Event(new StringEventType("a-=-d")));
        TrieNode<Event> root_ad_da = root_ad.getChild(new Event(new StringEventType("d-=-a")));
        TrieNode<Event> root_br = root.getChild(new Event(new StringEventType("b-=-r")));
        TrieNode<Event> root_br_ra = root_br.getChild(new Event(new StringEventType("r-=-a")));
        TrieNode<Event> root_ca = root.getChild(new Event(new StringEventType("c-=-a")));
        TrieNode<Event> root_ca_ad = root_ca.getChild(new Event(new StringEventType("a-=-d")));
        TrieNode<Event> root_da = root.getChild(new Event(new StringEventType("d-=-a")));
        TrieNode<Event> root_da_ab = root_da.getChild(new Event(new StringEventType("a-=-b")));
        TrieNode<Event> root_ra = root.getChild(new Event(new StringEventType("r-=-a")));
        TrieNode<Event> root_ra_ac = root_ra.getChild(new Event(new StringEventType("a-=-c")));
        TrieNode<Event> root_ra_aEnd = root_ra.getChild(new Event(new StringEventType("a-=-END")));
        TrieNode<Event> root_startA = root.getChild(new Event(new StringEventType("START-=-a")));
        TrieNode<Event> root_startA_ab =
            root_startA.getChild(new Event(new StringEventType("a-=-b")));

        assertEquals(1, root_aEnd.getCount());
        assertTrue(root_aEnd.isLeaf());
        assertEquals(2, root_ab.getCount());
        assertEquals(1, root_ab.getChildren().size());
        assertEquals(2, root_ab_br.getCount());
        assertTrue(root_ab_br.isLeaf());
        assertEquals(1, root_ac.getCount());
        assertEquals(1, root_ac.getChildren().size());
        assertEquals(1, root_ac_ca.getCount());
        assertTrue(root_ac_ca.isLeaf());
        assertEquals(1, root_ad.getCount());
        assertEquals(1, root_ad.getChildren().size());
        assertEquals(1, root_ad_da.getCount());
        assertTrue(root_ad_da.isLeaf());
        assertEquals(2, root_br.getCount());
        assertEquals(1, root_br.getChildren().size());
        assertEquals(2, root_br_ra.getCount());
        assertTrue(root_br_ra.isLeaf());
        assertEquals(1, root_ca.getCount());
        assertEquals(1, root_ca.getChildren().size());
        assertEquals(1, root_ca_ad.getCount());
        assertTrue(root_ca_ad.isLeaf());
        assertEquals(1, root_da.getCount());
        assertEquals(1, root_da.getChildren().size());
        assertEquals(1, root_da_ab.getCount());
        assertTrue(root_da_ab.isLeaf());
        assertEquals(2, root_ra.getCount());
        assertEquals(2, root_ra.getChildren().size());
        assertEquals(1, root_ra_ac.getCount());
        assertTrue(root_ra_ac.isLeaf());
        assertEquals(1, root_ra_aEnd.getCount());
        assertTrue(root_ra_aEnd.isLeaf());
        assertEquals(1, root_startA.getCount());
        assertEquals(1, root_startA.getChildren().size());
        assertEquals(1, root_startA_ab.getCount());
        assertTrue(root_startA_ab.isLeaf());
    }

    @Test
    public void testFlattenPredictionByPartialMatch_1() throws Exception {
        ModelFlattener fixture = new ModelFlattener();
        PredictionByPartialMatch model = new PredictionByPartialMatch(1, new Random());

        FirstOrderMarkovModel result = fixture.flattenPredictionByPartialMatch(model);

        assertEquals(null, result);
    }

    @Before
    public void setUp() throws Exception {
        sequence = new ArrayList<Event>();
        sequence.add(new Event(new StringEventType("a")));
        sequence.add(new Event(new StringEventType("b")));
        sequence.add(new Event(new StringEventType("r")));
        sequence.add(new Event(new StringEventType("a")));
        sequence.add(new Event(new StringEventType("c")));
        sequence.add(new Event(new StringEventType("a")));
        sequence.add(new Event(new StringEventType("d")));
        sequence.add(new Event(new StringEventType("a")));
        sequence.add(new Event(new StringEventType("b")));
        sequence.add(new Event(new StringEventType("r")));
        sequence.add(new Event(new StringEventType("a")));
    }

    public static void main(String[] args) {
        new org.junit.runner.JUnitCore().run(ModelFlattenerTest.class);
    }
}
