package de.ugoe.cs.quest.usageprofiles;

import java.security.InvalidParameterException;
import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Implements a round-trip buffered memory of a specified length that can be used to remember the
 * recent history. Every event that happend longer ago than the length of the memory is forgotten,
 * hence the memory is incomplete.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 * 
 * @param <T>
 *            Type which is memorized.
 */
public class IncompleteMemory<T> implements IMemory<T> {

    /**
     * <p>
     * Maximum length of the memory.
     * </p>
     */
    private int length;

    /**
     * <p>
     * Internal storage of the history.
     * </p>
     */
    private List<T> history;

    /**
     * <p>
     * Constructor. Creates a new IncompleteMemory.
     * </p>
     * 
     * @param length
     *            number of recent events that are remembered
     * @throws InvalidParameterException
     *             This exception is thrown if the length is smaller than 1
     */
    public IncompleteMemory(int length) {
        if (length < 1) {
            throw new InvalidParameterException("Length of IncompleteMemory must be at least 1.");
        }
        this.length = length;
        history = new LinkedList<T>();
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.quest.usageprofiles.IMemory#add(java.lang.Object)
     */
    @Override
    public void add(T state) {
        if (history.size() == length) {
            history.remove(0);
        }
        history.add(state);
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.quest.usageprofiles.IMemory#getLast(int)
     */
    @Override
    public List<T> getLast(int num) {
        if (num < 1) {
            return new LinkedList<T>();
        }
        else {
            return new LinkedList<T>(history.subList(Math.max(0, history.size() - num),
                                                     history.size())); // defensive copy
        }
    }

    /**
     * <p>
     * Returns the current length of the memory. This can be less than {@link #length}, if the
     * overall history is less than {@link #length}.
     * </p>
     * 
     * @return length of the current memory
     */
    public int getLength() {
        return history.size();
    }
}
