
package de.ugoe.cs.quest.commands.sequences;

import java.security.InvalidParameterException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import de.ugoe.cs.quest.CommandHelpers;
import de.ugoe.cs.quest.SequenceInstanceOf;
import de.ugoe.cs.quest.eventcore.Event;
import de.ugoe.cs.quest.eventcore.gui.SortedInteractionEventList;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to sort the key interactions in a sequence of events. An example, the sequence
 * to write the upper case D
 * <ul>
 *   <li>press shift key</li>
 *   <li>press D key</li>
 *   <li>release shift key</li>
 *   <li>release D key</li>
 * </ul>
 * 
 * is transformed to the sequence
 * 
 * <ul>
 *   <li>press shift key</li>
 *   <li>press D key</li>
 *   <li>release D key</li>
 *   <li>release shift key</li>
 * </ul>
 * 
 * in which the first pressed key (shift in this case) is always released last. The same is done
 * for the alt and the ctrl keys.
 * 
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDsortKeyInteractions implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "sortKeyInterations <sequences> {<new sequences>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String sequencesName;
        String newSequencesName;
        try {
            sequencesName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                newSequencesName = (String) parameters.get(1);
            }
            else {
                newSequencesName = sequencesName;
            }
        }
        catch (Exception e) {
            throw new InvalidParameterException("must provide a sequences name");
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;
        SortedInteractionEventList sortedEventList;

        Collection<List<Event>> newSequences = new LinkedList<List<Event>>();
        
        for (List<Event> sequence : sequences) {
            sortedEventList = new SortedInteractionEventList();
            
            for (Event event : sequence) {
                sortedEventList.add(event);
            }
            
            newSequences.add(sortedEventList);
        }

        if (GlobalDataContainer.getInstance().addData(newSequencesName, newSequences)) {
            CommandHelpers.dataOverwritten(sequencesName);
        }
        
    }

}
