
package de.ugoe.cs.quest.commands.sequences;

import java.security.InvalidParameterException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.quest.CommandHelpers;
import de.ugoe.cs.quest.SequenceInstanceOf;
import de.ugoe.cs.quest.eventcore.Event;
import de.ugoe.cs.quest.eventcore.gui.KeyInteractionSorter;
import de.ugoe.cs.quest.eventcore.gui.KeyInteractionSorter.CleanupMode;
import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;
import de.ugoe.cs.util.console.GlobalDataContainer;

/**
 * <p>
 * Command to sort the key interactions in a sequence of events. An example, the sequence to write
 * the upper case D
 * <ul>
 * <li>press shift key</li>
 * <li>press D key</li>
 * <li>release shift key</li>
 * <li>release D key</li>
 * </ul>
 * 
 * is transformed to the sequence
 * 
 * <ul>
 * <li>press shift key</li>
 * <li>press D key</li>
 * <li>release D key</li>
 * <li>release shift key</li>
 * </ul>
 * 
 * in which the first pressed key (shift in this case) is always released last. The same is done for
 * the alt and the ctrl keys.
 * 
 * </p>
 * 
 * @author Patrick Harms
 * @version 1.0
 */
public class CMDsortKeyInteractions implements Command {

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#help()
     */
    @Override
    public String help() {
        return "sortKeyInterations <sequences> {<new sequences>}";
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.util.console.Command#run(java.util.List)
     */
    @SuppressWarnings("unchecked")
    @Override
    public void run(List<Object> parameters) {
        String sequencesName;
        String newSequencesName;
        String modeString = "ADDITION";
        try {
            sequencesName = (String) parameters.get(0);
            if (parameters.size() > 1) {
                newSequencesName = (String) parameters.get(1);
            }
            else {
                newSequencesName = sequencesName;
            }
            if (parameters.size() > 2) {
                modeString = (String) parameters.get(2);
            }
        }
        catch (Exception e) {
            throw new InvalidParameterException("must provide a sequences name");
        }

        Collection<List<Event>> sequences = null;
        Object dataObject = GlobalDataContainer.getInstance().getData(sequencesName);
        if (dataObject == null) {
            CommandHelpers.objectNotFoundMessage(sequencesName);
            return;
        }
        if (!SequenceInstanceOf.isCollectionOfSequences(dataObject)) {
            CommandHelpers.objectNotType(sequencesName, "Collection<List<Event<?>>>");
            return;
        }

        CleanupMode mode = null;
        try {
            mode = CleanupMode.valueOf(modeString);
        }
        catch (IllegalArgumentException e) {
            Console.printerrln("Invalid mode. Only REMOVAL and ADDITION are allowed values!");
            return;
        }

        sequences = (Collection<List<Event>>) dataObject;

        Collection<List<Event>> newSequences = new LinkedList<List<Event>>();

        int i = 1;
        for (List<Event> sequence : sequences) {
            Console.traceln(Level.INFO, "Processing sequence " + i++);
            newSequences.add(new KeyInteractionSorter(mode).sortKeyInteractions(sequence));
        }

        if (GlobalDataContainer.getInstance().addData(newSequencesName, newSequences)) {
            CommandHelpers.dataOverwritten(newSequencesName);
        }

    }

}
