package de.ugoe.cs.quest.plugin.jfc.commands;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.logging.Level;

import de.ugoe.cs.util.console.Command;
import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * Command to pre-process files written by EventBench's JFCMonitor located in a
 * directory. The only task of the pre-processing is checking if the session was
 * closed properly, i.e., if the XML file ends with a {@code </sessions>} tag.
 * If this is not the case, the tag will be appended to the file.
 * </p>
 * 
 * @author Steffen Herbold
 * @version 1.0
 */
public class CMDpreprocessDirJFC implements Command {

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#run(java.util.List)
	 */
	@Override
	public void run(List<Object> parameters) {
		String sourcePath;
		String targetPath;
		try {
			sourcePath = (String) parameters.get(0);
			targetPath = (String) parameters.get(1);
		} catch (Exception e) {
			throw new IllegalArgumentException();
		}

		File sourceFolder = new File(sourcePath);
		if (!sourceFolder.isDirectory()) {
			Console.printerrln(sourcePath + " is not a directory");
		}
		String absolutPathSource = sourceFolder.getAbsolutePath();
		File targetFolder = new File(targetPath);
		if (!targetFolder.isDirectory()) {
			Console.printerrln(targetPath + " is not a directory");
		}
		String absolutPathTarget = targetFolder.getAbsolutePath();

		for (String filename : sourceFolder.list()) {
			String source = absolutPathSource + "/" + filename;
			Console.traceln(Level.INFO, "Preprocessing file: " + source);
			File file = new File(source);
			InputStreamReader reader;
			try {
				FileInputStream fis = new FileInputStream(file);
				reader = new InputStreamReader(fis, "UTF-16");
			} catch (FileNotFoundException e) {
				Console.printerrln(e.getMessage());
				return;
			} catch (UnsupportedEncodingException e) {
				Console.printerrln(e.getMessage());
				return;
			}
			char[] buffer = new char[(int) file.length()];
			try {
				reader.read(buffer);
				reader.close();
			} catch (IOException e) {
				Console.printerrln(e.getMessage());
				return;
			}

			String content = new String(buffer).trim();

			int index = filename.lastIndexOf('.');
			String target = absolutPathTarget + "/"
					+ filename.substring(0, index) + ".xml";

			Console.traceln(Level.INFO, "   Saving as: " + target);

			OutputStreamWriter writer;
			try {
				FileOutputStream fos = new FileOutputStream(target);
				writer = new OutputStreamWriter(fos, "UTF-8");
			} catch (IOException e) {
				Console.printerrln(e.getMessage());
				return;
			}
			try {
				writer.write(content);
				if (!content.endsWith("</sessions>")) {
					writer.write("</sessions>");
				}
				writer.close();
			} catch (IOException e) {
				Console.printerrln(e.getMessage());
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.ugoe.cs.util.console.Command#help()
	 */
	@Override
	public String help() {
		return "preprocessDirJFC <sourceDirectory> <targetDirectory>";
	}

}
