
package de.ugoe.cs.quest.plugin.jfc.guimodel;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections15.CollectionUtils;

import de.ugoe.cs.quest.eventcore.guimodel.IGUIElement;
import de.ugoe.cs.quest.eventcore.guimodel.IGUIElementSpec;

/**
 * <p>
 * Implements the specification of {@link IGUIElement} for {@link JFCGUIElement}s.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms
 */
public class JFCGUIElementSpec implements IGUIElementSpec {

    /**
     * <p>
     * Id for object serialization.
     * </p>
     */
    private static final long serialVersionUID = 1L;

    /**
     * <p>
     * Current name of the GUI element
     * </p>
     */
    private String name;

    /**
     * <p>
     * Previous names of the GUI element as it may have changed over time.
     * </p>
     */
    private List<String> formerNames = new ArrayList<String>();

    /**
     * <p>
     * Type of the GUI element, i.e., its Java class.
     * </p>
     */
    private String type = null;

    /**
     * <p>
     * Icon associated with the GUI element.
     * </p>
     */
    private String icon = null;

    /**
     * <p>
     * Index of the GUI element in its parent element.
     * </p>
     */
    private int index = -1;

    /**
     * <p>
     * Hash code of the window element. Used as unique identifier during its existence.
     * </p>
     */
    private int elementHash = -1;

    /**
     * <p>
     * Previous hashes of the window as the window may have been destroyed and recreated.
     * </p>
     */
    private List<Integer> formerElementHashes = new ArrayList<Integer>();

    /*
     * (non-Javadoc)
     * 
     * @see
     * de.ugoe.cs.quest.eventcore.guimodel.IGUIElementSpec#getSecificationSimilarity(IGUIElementSpec
     * )
     */
    @Override
    public boolean getSimilarity(IGUIElementSpec other) {
        if (this == other) {
            return true;
        }

        if (!(other instanceof JFCGUIElementSpec)) {
            return false;
        }

        JFCGUIElementSpec otherSpec = (JFCGUIElementSpec) other;

        if ((type != otherSpec.type) && ((type != null) && (!type.equals(otherSpec.type)))) {
            return false;
        }

        if ((icon != otherSpec.icon) && ((icon != null) && (!icon.equals(otherSpec.icon)))) {
            return false;
        }

        // up to now, we compared, if the basics match. Now lets compare the id, the name and the
        // index. All may change. The name may be reset (e.g. the title of a frame using the
        // asterisk in the case data was changed). The id may change if e.g. a dialog is closed
        // and reopend, i.e. a new instance is created. The index may change, if later in a panel
        // a new element is added or another one is removed. If the element hash or the name stay
        // the same, then similarity is given. Therefore these are the first two comparisons

        if (elementHash == otherSpec.elementHash) {
            return true;
        }

        if ((name != null) && (name.equals(otherSpec.name))) {
            return true;
        }

        if ((((name == null) && (otherSpec.name == null)) || (("".equals(name)) && (""
            .equals(otherSpec.name)))) &&
            (formerNames.size() == 0) &&
            (otherSpec.formerNames.size() == 0))
        {
            return true;
        }

        // if the id and the name did not stay the same, then the name should be checked first.
        // One of all known names of one of the specs must be equal to one of the known names of the
        // respective other spec for similarity. Furthermore, if this is given, the index should
        // have stayed the same.

        if ((otherSpec.name != null) && formerNames.contains(otherSpec.name)) {
            return index == otherSpec.index;
        }

        if ((name != null) && otherSpec.formerNames.contains(name)) {
            return index == otherSpec.index;
        }

        if (CollectionUtils.containsAny(formerNames, otherSpec.formerNames)) {
            return index == otherSpec.index;
        }

        // ok. Even the names do not match. This is usually a clear indication, that the elements
        // are distinct. However, we check, if the former ids matched. This is very unlikely
        // to happen. But it may occur, if a GUI element does not have a name or its name stays
        // the empty string and if this GUI element is created, destroyed, and created again.
        // Again we are restrictive and request the index to be equal as well.

        if (formerElementHashes.contains(otherSpec.elementHash)) {
            return index == otherSpec.index;
        }

        if (otherSpec.formerElementHashes.contains(elementHash)) {
            return index == otherSpec.index;
        }

        if (CollectionUtils.containsAny(formerElementHashes, otherSpec.formerElementHashes)) {
            return index == otherSpec.index;
        }

        // now we can be really sure, that the GUI elements differ

        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.ugoe.cs.quest.eventcore.guimodel.IGUIElementSpec#equals(IGUIElementSpec)
     */
    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }

        if (!(other instanceof JFCGUIElementSpec)) {
            return false;
        }

        JFCGUIElementSpec otherSpec = (JFCGUIElementSpec) other;

        return ((name == otherSpec.name) || ((name != null) && (name.equals(otherSpec.name)))) &&
            ((type == otherSpec.type) || ((type != null) && (type.equals(otherSpec.type)))) &&
            ((icon == otherSpec.icon) || ((icon != null) && (icon.equals(otherSpec.icon)))) &&
            (index == otherSpec.index) && (elementHash == otherSpec.elementHash);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        return (name + type + icon + index + elementHash).hashCode();
    }

    /**
     * <p>
     * Returns the name of the specified GUI element.
     * </p>
     * 
     * @return the name
     */
    public String getName() {
        StringBuffer names = new StringBuffer();

        if (name != null) {
            names.append('"');
            names.append(name);
            names.append('"');
        }
        else {
            names.append("NOT_SET");
        }

        if (formerNames.size() > 0) {

            names.append(" (aka ");

            for (int i = 0; i < formerNames.size(); i++) {
                if (i > 0) {
                    names.append("/");
                }

                names.append('"');
                names.append(formerNames.get(i));
                names.append('"');
            }

            names.append(")");
        }

        return names.toString();
    }

    /**
     * <p>
     * Returns the title of the specified GUI element.
     * </p>
     * 
     * @return the title
     */
    public String getType() {
        return type;
    }

    /**
     * <p>
     * Returns the icon associated with the specified GUI element.
     * </p>
     * 
     * @return the icon
     */
    public String getIcon() {
        return icon;
    }

    /**
     * <p>
     * Returns the index of the specified GUI element in its parent element.
     * </p>
     * 
     * @return the index
     */
    public int getIndex() {
        return index;
    }

    /**
     * <p>
     * Returns the object hash of the specified GUI element.
     * </p>
     * 
     * @return the elementHash
     */
    public int getElementHash() {
        return elementHash;
    }

    /**
     * <p>
     * Sets the name of the specified GUI element.
     * </p>
     * 
     * @param newName
     *            the name
     */
    public void setName(String newName) {
        if ((this.name != null) && (!this.name.equals(newName)) &&
            (!this.formerNames.contains(this.name)))
        {
            this.formerNames.add(this.name);
        }

        this.name = newName;
    }

    /**
     * <p>
     * Sets the type of the specified GUI element.
     * </p>
     * 
     * @param type
     *            the type
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * <p>
     * Sets the icon associated with the specified GUI element.
     * </p>
     * 
     * @param icon
     *            the icon
     */
    public void setIcon(String icon) {
        this.icon = icon;
    }

    /**
     * <p>
     * Sets the index in its parent element of the specified GUI element.
     * </p>
     * 
     * @param index
     *            the index
     */
    public void setIndex(int index) {
        this.index = index;
    }

    /**
     * <p>
     * Sets the object hash of the specified GUI element.
     * </p>
     * 
     * @param newElementHash
     *            the element hash
     */
    public void setElementHash(int newElementHash) {
        if ((this.elementHash > -1) && !this.formerElementHashes.contains(this.elementHash)) {
            this.formerElementHashes.add(this.elementHash);
        }

        this.elementHash = newElementHash;
    }

    /**
     * <p>
     * Updates the specification with another specification.
     * </p>
     * 
     * @param furtherSpec
     *            specification used to update the current specification
     */
    void update(JFCGUIElementSpec other) {
        if (other != this) {
            for (int formerElementHash : other.formerElementHashes) {
                setElementHash(formerElementHash);
            }

            if (elementHash != other.elementHash) {
                elementHash = other.elementHash;
            }

            for (String formerName : other.formerNames) {
                setName(formerName);
            }

            if ((name != other.name) && (name != null) && (!name.equals(other.name))) {
                setName(other.name);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return "[" + getName() + ";\"" + type + "\";\"" + icon + "\";" + index + ";" + elementHash +
            "]";
    }

}
