package de.ugoe.cs.autoquest.htmlmonitor;

import de.ugoe.cs.util.console.Console;

/**
 * <p>
 * The HTML monitor starts a web server ({@link HtmlMonitorServer}) that receives log messages
 * of the HTML monitor java script. These messages are logged using the
 * {@link HtmlMonitorLogManager}. The class assures that on shutdown e.g. caused by CTRL-C the
 * server and the log manager are stopped correctly.
 * </p>
 * 
 * @author Patrick Harms
 */
public class HtmlMonitor implements HtmlMonitorComponent {

    /**
     * the port on which the webserver shall listen. Defaults to 8090.
     */
    private int port = 8090;
    
    /**
     * the web server receiving the log messages
     */
    private HtmlMonitorServer server;
    
    /**
     * the directory into which the log files shall be written
     */
    private String logFileBaseDir;

    /**
     * the log manager being responsible for performing the logging
     */
    private HtmlMonitorLogManager logManager;

    /**
     * the thread needed to handle CTRL-C events
     */
    private Thread shutdownHook;

    /**
     * <p>
     * initializes the monitor with the command line arguments. Those may be the log directory
     * as first argument and the port to listen on as second
     * </p>
     *
     * @param commandLineArguments the command line arguments when starting the monitor using
     *                             the {@link Runner}
     */
    public HtmlMonitor(String[] commandLineArguments) {
        if (commandLineArguments.length > 0) {
            this.logFileBaseDir = commandLineArguments[0];
            Console.println("putting logs into directory " + this.logFileBaseDir);
        }
        
        if (commandLineArguments.length > 1) {
            try {
                this.port = Integer.parseInt(commandLineArguments[1]);
            }
            catch (NumberFormatException e) {
                Console.println("ignoring invalid port specification " + commandLineArguments[1]);
            }
            Console.println("listening on port " + this.port);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HtmlMonitorComponent#init()
     */
    @Override
    public synchronized void init() throws HtmlMonitorException {
        if (server != null) {
            throw new IllegalStateException("already initialized.");
        }
        
        try {
            logManager = new HtmlMonitorLogManager(logFileBaseDir);
            logManager.init();
        
            server = new HtmlMonitorServer(port, logManager);
            server.init();
        
            shutdownHook = new Thread(new ShutdownHook(server, logManager));
        }
        catch (HtmlMonitorException e) {
            Console.printerrln("could not initialize HTML monitor: " + e);
            Console.logException(e);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HtmlMonitorComponent#start()
     */
    @Override
    public synchronized void start() {
        if (server == null) {
            throw new IllegalStateException("not initialized.");
        }
        
        try {
            Runtime.getRuntime().addShutdownHook(shutdownHook);
            logManager.start();
            server.start();
        }
        catch (HtmlMonitorException e) {
            Console.printerrln("could not start HTML monitor: " + e);
            Console.logException(e);
        }
    }

    /* (non-Javadoc)
     * @see de.ugoe.cs.autoquest.htmlmonitor.HtmlMonitorComponent#stop()
     */
    @Override
    public synchronized void stop() {
        if (server == null) {
            throw new IllegalStateException("not initialized.");
        }
        
        Runtime.getRuntime().removeShutdownHook(shutdownHook);
        server.stop();
        logManager.stop();
        
        server = null;
        logManager = null;
    }

}
