
package de.ugoe.cs.quest.plugin.mfc.eventcore;

/**
 * <p>
 * Enumeration to deal with MFC message types.
 * </p>
 * 
 * @version 1.0
 * @author Patrick Harms, Steffen Herbold
 */
public enum WindowsMessageType {

    WM_NULL(0), WM_CREATE(1), WM_DESTROY(2), WM_MOVE(3), WM_SIZE(5), WM_ACTIVATE(6),
    WM_SETFOCUS(7), WM_KILLFOCUS(8), WM_ENABLE(10), WM_SETREDRAW(11), WM_SETTEXT(12),
    WM_GETTEXT(13), WM_GETTEXTLENGTH(14), WM_PAINT(15), WM_CLOSE(16), WM_QUERYENDSESSION(17),
    WM_QUIT(18), WM_QUERYOPEN(19), WM_ERASEBKGND(20), WM_SYSCOLORCHANGE(21), WM_ENDSESSION(22),
    WM_SHOWWINDOW(24), WM_CTLCOLOR(25), WM_WININICHANGE(26), WM_DEVMODECHANGE(27), WM_ACTIVATEAPP(
        28), WM_FONTCHANGE(29), WM_TIMECHANGE(30), WM_CANCELMODE(31), WM_SETCURSOR(32),
    WM_MOUSEACTIVATE(33), WM_CHILDACTIVATE(34), WM_QUEUESYNC(35), WM_GETMINMAXINFO(36),
    WM_PAINTICON(38), WM_ICONERASEBKGND(39), WM_NEXTDLGCTL(40), WM_SPOOLERSTATUS(42), WM_DRAWITEM(
        43), WM_MEASUREITEM(44), WM_DELETEITEM(45), WM_VKEYTOITEM(46), WM_CHARTOITEM(47),
    WM_SETFONT(48), WM_GETFONT(49), WM_SETHOTKEY(50), WM_GETHOTKEY(51), WM_QUERYDRAGICON(55),
    WM_COMPAREITEM(57), WM_GETOBJECT(61), WM_COMPACTING(65), WM_COMMNOTIFY(68),
    WM_WINDOWPOSCHANGING(70), WM_WINDOWPOSCHANGED(71), WM_POWER(72), WM_COPYDATA(74),
    WM_CANCELJOURNAL(75), WM_NOTIFY(78), WM_INPUTLANGCHANGEREQUEST(80), WM_INPUTLANGCHANGE(81),
    WM_TCARD(82), WM_HELP(83), WM_USERCHANGED(84), WM_NOTIFYFORMAT(85), WM_CONTEXTMENU(123),
    WM_STYLECHANGING(124), WM_STYLECHANGED(125), WM_DISPLAYCHANGE(126), WM_GETICON(127),
    WM_SETICON(128), WM_NCCREATE(129), WM_NCDESTROY(130), WM_NCCALCSIZE(131), WM_NCHITTEST(132),
    WM_NCPAINT(133), WM_NCACTIVATE(134), WM_GETDLGCODE(135), WM_SYNCPAINT(136),
    WM_NCMOUSEMOVE(160), WM_NCLBUTTONDOWN(161), WM_NCLBUTTONUP(162), WM_NCLBUTTONDBLCLK(163),
    WM_NCRBUTTONDOWN(164), WM_NCRBUTTONUP(165), WM_NCRBUTTONDBLCLK(166), WM_NCMBUTTONDOWN(167),
    WM_NCMBUTTONUP(168), WM_NCMBUTTONDBLCLK(169), WM_NCXBUTTONDOWN(171), WM_NCXBUTTONUP(172),
    WM_NCXBUTTONDBLCLK(173), SBM_SETPOS(224), BM_CLICK(245), WM_INPUT(255), WM_KEYDOWN(256),
    WM_KEYFIRST(256), WM_KEYUP(257), WM_CHAR(258), WM_DEADCHAR(259), WM_SYSKEYDOWN(260),
    WM_SYSKEYUP(261), WM_SYSCHAR(262), WM_SYSDEADCHAR(263), WM_KEYLAST(264),
    WM_WNT_CONVERTREQUESTEX(265), WM_CONVERTREQUEST(266), WM_CONVERTRESULT(267), WM_INTERIM(268),
    WM_IME_STARTCOMPOSITION(269), WM_IME_ENDCOMPOSITION(270), WM_IME_COMPOSITION(271),
    WM_IME_KEYLAST(271), WM_INITDIALOG(272), WM_COMMAND(273), WM_SYSCOMMAND(274), WM_TIMER(275),
    WM_HSCROLL(276), WM_VSCROLL(277), WM_INITMENU(278), WM_INITMENUPOPUP(279), WM_MENUSELECT(287),
    WM_MENUCHAR(288), WM_ENTERIDLE(289), WM_MENURBUTTONUP(290), WM_MENUDRAG(291), WM_MENUGETOBJECT(
        292), WM_UNINTMENUPOPUP(293), WM_MENUCOMMAND(294), WM_CHANGEUISTATE(295), WM_UPDATEUISTATE(
        296), WM_QUERYUISTATE(297), WM_CTLCOLORMSGBOX(306), WM_CTLCOLOREDIT(307),
    WM_CTLCOLORLISTBOX(308), WM_CTLCOLORBTN(309), WM_CTLCOLORDLG(310), WM_CTLCOLORSCROLLBAR(311),
    WM_CTLCOLORSTATIC(312), CB_SHOWDROPDOWN(335), LB_SETCURSEL(390), WM_MOUSEFIRST(512),
    WM_MOUSEMOVE(512), WM_LBUTTONDOWN(513), WM_LBUTTONUP(514), WM_LBUTTONDBLCLK(515),
    WM_RBUTTONDOWN(516), WM_RBUTTONUP(517), WM_RBUTTONDBLCLK(518), WM_MBUTTONDOWN(519),
    WM_MBUTTONUP(520), WM_MBUTTONDBLCLK(521), WM_MOUSELAST(521), WM_MOUSEWHEEL(522),
    WM_XBUTTONDOWN(523), WM_XBUTTONUP(524), WM_XBUTTONDBLCLK(525), WM_USER(1024),
    CB_SETCURSEL(334), TBM_SETPOS(1029), UDM_SETRANGE(1125), TCM_SETCURSEL(4876);

    /**
     * <p>
     * Numerical representation of the message type.
     * </p>
     */
    private int mNumber;

    /**
     * <p>
     * Constructor. Creates a new WindowsMessageType.
     * </p>
     * 
     * @param number
     *            numerical representation of the message type
     */
    WindowsMessageType(int number) {
        mNumber = number;
    }

    /**
     * <p>
     * Returns the numerical representation of the message type.
     * </p>
     * 
     * @return the numerical representation
     */
    public int getNumber() {
        return mNumber;
    }

    /**
     * <p>
     * Checks if the type of a message generated is a keyboard interaction.
     * </p>
     * 
     * @param msgType
     *            type of the message
     * @return true if it is a keyboard interaction; false otherwise
     */
    public boolean isKeyMessage() {
        boolean isKeyMsg = false;
        switch (this)
        {
            case WM_KEYDOWN:
            case WM_KEYUP:
            case WM_SYSKEYDOWN:
            case WM_SYSKEYUP:
                isKeyMsg = true;
                break;
            default:
                break;
        }
        return isKeyMsg;
    }

    /**
     * <p>
     * Checks if the type of a message indicates that the mouse has been pressed down.
     * </p>
     * 
     * @param msgType
     *            type of the message
     * @return true if it is mouse-down message; false otherwise
     */
    public boolean isDownMessage() {
        boolean isDownMsg = false;
        switch (this)
        {
            case WM_LBUTTONDOWN:
            case WM_RBUTTONDOWN:
            case WM_MBUTTONDOWN:
            case WM_XBUTTONDOWN:
            case WM_NCLBUTTONDOWN:
            case WM_NCRBUTTONDOWN:
            case WM_NCMBUTTONDOWN:
            case WM_NCXBUTTONDOWN:
                isDownMsg = true;
                break;
            default:
                break;
        }
        return isDownMsg;
    }

    /**
     * <p>
     * Checks if the type of a message indicates that a double click has been performed.
     * </p>
     * 
     * @param msgType
     *            type of the message
     * @return true if it is a double click message; false otherwise
     */
    public boolean isDblclkMessage() {
        boolean isDblclkMsg = false;
        switch (this)
        {
            case WM_LBUTTONDBLCLK:
            case WM_RBUTTONDBLCLK:
            case WM_MBUTTONDBLCLK:
            case WM_XBUTTONDBLCLK:
            case WM_NCLBUTTONDBLCLK:
            case WM_NCRBUTTONDBLCLK:
            case WM_NCMBUTTONDBLCLK:
            case WM_NCXBUTTONDBLCLK:
                isDblclkMsg = true;
                break;
            default:
                break;
        }
        return isDblclkMsg;
    }

    /**
     * <p>
     * Checks if the type of a message indicates that the mouse has been released.
     * </p>
     * 
     * @param msgType
     *            type of the message
     * @return true if it is mouse-up message; false otherwise
     */
    public boolean isUpMessage() {
        boolean isUpMsg = false;
        switch (this)
        {
            case WM_LBUTTONUP:
            case WM_RBUTTONUP:
            case WM_MBUTTONUP:
            case WM_XBUTTONUP:
            case WM_NCLBUTTONUP:
            case WM_NCRBUTTONUP:
            case WM_NCMBUTTONUP:
            case WM_NCXBUTTONUP:
                isUpMsg = true;
                break;
            default:
                break;
        }
        return isUpMsg;
    }

    /**
     * <p>
     * Parses an {@link String} and returns the respective WindowsMessageType if possible.
     * </p>
     * 
     * @param numberString
     *            String representation of the event type
     * @return created WindowsMessageType
     * @throws IllegalArgumentException
     *             thrown if there is no WindowsMessageType that correlates to numberString
     */
    public static WindowsMessageType parseMessageType(String numberString)
        throws IllegalArgumentException
    {
        try {
            int number = Integer.parseInt(numberString);
            return valueOf(number);
        }
        catch (NumberFormatException e) {
            return WindowsMessageType.valueOf(WindowsMessageType.class, numberString);
        }
    }

    /**
     * <p>
     * Returns the WindowsMessageType associated with an integer.
     * </p>
     * 
     * @param number
     *            integer to which the according WindowsMessageType is returned
     * @return the WindowsMessageType
     * @throws IllegalArgumentException
     *             thrown if there is no WindowsMessageType that correlates to number
     */
    public static WindowsMessageType valueOf(int number) throws IllegalArgumentException {
        for (WindowsMessageType type : WindowsMessageType.values()) {
            if (type.mNumber == number) {
                return type;
            }
        }

        throw new IllegalArgumentException("there is no message type with number " + number);
    }
}
